/*
 * Tentackle - https://tentackle.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.tentackle.model.impl;

import org.tentackle.common.Service;
import org.tentackle.common.StringHelper;
import org.tentackle.model.Attribute;
import org.tentackle.model.Entity;
import org.tentackle.model.Index;
import org.tentackle.model.NameVerifier;
import org.tentackle.model.Relation;

import java.util.Locale;
import java.util.Set;

/**
 * Default implementation of a {@link NameVerifier}.
 */
@Service(NameVerifier.class)
public class NameVerifierImpl implements NameVerifier {

  private static final String TABLE_NAME = "table name '";
  private static final String TABLE_ALIAS = "table alias '";
  private static final String COLUMN_NAME = "column name '";

  private static final String MIXED_UC_LC = "' contains mixed upper- and lowercase characters";
  private static final String QUOTED = "' is quoted, which is only allowed for [PROVIDED] entities";

  // generated by IllegalModelNamesTest (tentackle-test-pdo module)
  private static final Set<String> ILLEGAL_NAMES = Set.of(
    "abstract",
    "backend",
    "baseContext",
    "cache",
    "cacheAccessCount",
    "cacheAccessTime",
    "cacheable",
    "cached",
    "class",
    "classBaseName",
    "classId",
    "classVariables",
    "columnCount",
    "columnName",
    "columnNames",
    "columnPrefix",
    "composite",
    "contextId",
    "contextUserId",
    "copy",
    "countingModification",
    "defaultScopes",
    "deleted",
    "domainContext",
    "domainContextImmutable",
    "domainDelegate",
    "eagerJoinedSelect",
    "eagerJoins",
    "editAllowed",
    "editedBy",
    "editedExpiry",
    "editedSince",
    "effectiveClass",
    "effectiveSuperClasses",
    "embedded",
    "embeddedFields",
    "embeddingParent",
    "embeddingPersistentObject",
    "entity",
    "expirationBacklog",
    "expired",
    "expiredTableSerials",
    "explicitIdAliasRequiredInJoins",
    "fields",
    "finallyImmutable",
    "forcedModified",
    "fromThisJVM",
    "id",
    "idSource",
    "idValid",
    "immutable",
    "immutableLoggingLevel",
    "loggingModification",
    "modificationCount",
    "modificationLog",
    "modified",
    "new",
    "normText",
    "normTextProvided",
    "overloadable",
    "pdo",
    "pdoClass",
    "pdoMethodCache",
    "permissionAccepted",
    "persistable",
    "persistenceClass",
    "preparedStatement",
    "readAllowed",
    "referenced",
    "remoteDelegate",
    "removable",
    "renewTokenLockRequested",
    "replayedLeniently",
    "rootClassId",
    "rootClassIdProvided",
    "rootEntity",
    "rootEntityOf",
    "rootId",
    "rootIdProvided",
    "securityResult",
    "serial",
    "session",
    "sessionHolder",
    "sessionImmutable",
    "snapshot",
    "snapshots",
    "sqlClassIdCondition",
    "sqlContextCondition",
    "statementAlwaysPrepared",
    "tableAlias",
    "tableName",
    "tableSerial",
    "tableSerialProvided",
    "tokenLockProvided",
    "tokenLockTimeout",
    "tokenLockableByMe",
    "tokenLocked",
    "tokenLockedBy",
    "tokenLockedByMe",
    "topSuperTableAlias",
    "topSuperTableName",
    "tracked",
    "transientData",
    "updatingSerialEvenIfNotModified",
    "validClassIds",
    "validated",
    "viewAllowed",
    "virgin",
    "writeAllowed"
  );

  private static final Set<String> CONFUSING_NAMES = Set.of(
    "ABSTRACT",
    "ACCEPTPERSISTENCEVISITOR",
    "ADDCOMPONENTS",
    "ADDPROPERTYLISTENER",
    "ADDREFERENCINGCLASS",
    "APPLYTOKENLOCKINFO",
    "ASSERTNOTABSTRACT",
    "ASSERTROOTCONTEXTISACCEPTED",
    "ASSERTROOTENTITY",
    "ATTRIBUTESMODIFIED",
    "BACKEND",
    "BASECONTEXT",
    "CACHE",
    "CACHEABLE",
    "CACHEACCESSCOUNT",
    "CACHEACCESSTIME",
    "CACHED",
    "CLASS",
    "CLASSBASENAME",
    "CLASSID",
    "CLASSVARIABLES",
    "CLEARONREMOTESAVE",
    "CLEARTOKENLOCK",
    "COLUMNCOUNT",
    "COLUMNNAME",
    "COLUMNNAMES",
    "COLUMNPREFIX",
    "COMPARETO",
    "COMPOSITE",
    "CONFIGUREEMBEDDEDCOLUMNS",
    "CONFIGUREREMOTEOBJECT",
    "CONFIGUREREMOTEOBJECTS",
    "CONTAINSPATTERN",
    "CONTEXTID",
    "CONTEXTUSERID",
    "COPY",
    "COUNTINGMODIFICATION",
    "COUNTMODIFICATION",
    "CREATEATTRIBUTESNORMTEXT",
    "CREATEDELETEALLSQL",
    "CREATEDELETESQL",
    "CREATEDUMMYUPDATESQL",
    "CREATEINSERTSQL",
    "CREATEMODIFICATIONLOG",
    "CREATEPREPAREDSTATEMENT",
    "CREATEQUERY",
    "CREATERELATIONSNORMTEXT",
    "CREATESELECTALLBYIDINNERSQL",
    "CREATESELECTALLIDSERIALINNERSQL",
    "CREATESELECTALLINNERSQL",
    "CREATESELECTALLSQL",
    "CREATESELECTALLWITHEXPIREDTABLESERIALSSQL",
    "CREATESELECTBYNORMTEXTSQL",
    "CREATESELECTEXPIREDTABLESERIALS1SQL",
    "CREATESELECTEXPIREDTABLESERIALS2SQL",
    "CREATESELECTIDINNERSQL",
    "CREATESELECTMAXIDSQL",
    "CREATESELECTMAXTABLESERIALSQL",
    "CREATESELECTOBJECTSWITHEXPIREDTABLESERIALSSQL",
    "CREATESELECTSERIALSQL",
    "CREATESELECTSQL",
    "CREATESELECTTOKENLOCKSQL",
    "CREATESNAPSHOT",
    "CREATESQLUPDATE",
    "CREATETRANSFERTOKENLOCKSQL",
    "CREATETRANSFERTOKENLOCKWITHTABLESERIALSQL",
    "CREATEUPDATEANDSETSERIALSQL",
    "CREATEUPDATESERIALANDTABLESERIALSQL",
    "CREATEUPDATESERIALSQL",
    "CREATEUPDATESQL",
    "CREATEUPDATETOKENLOCKONLYSQL",
    "CREATEUPDATETOKENLOCKSQL",
    "CREATEUPDATETOKENLOCKWITHCOUNTSQL",
    "CREATEVALIDCONTEXT",
    "DEFAULTSCOPES",
    "DELETE",
    "DELETED",
    "DELETEOBJECT",
    "DELETEPLAIN",
    "DELETEPLAINWITHCOMPONENTS",
    "DELETEREFERENCEDRELATIONS",
    "DELETEREFERENCINGRELATIONS",
    "DERIVEPDOFROMPO",
    "DETERMINECONTEXTID",
    "DIFFERSPERSISTED",
    "DISCARDSNAPSHOT",
    "DISCARDSNAPSHOTS",
    "DOMAINCONTEXT",
    "DOMAINCONTEXTIMMUTABLE",
    "DOMAINDELEGATE",
    "DUMMYUPDATE",
    "EAGERJOINEDSELECT",
    "EAGERJOINS",
    "EDITALLOWED",
    "EDITEDBY",
    "EDITEDEXPIRY",
    "EDITEDSINCE",
    "EFFECTIVECLASS",
    "EFFECTIVESUPERCLASSES",
    "EMBEDDED",
    "EMBEDDEDFIELDS",
    "EMBEDDINGPARENT",
    "EMBEDDINGPERSISTENTOBJECT",
    "ENTITY",
    "EQUALS",
    "EXECUTEFIRSTPDOQUERY",
    "EXECUTELISTQUERY",
    "EXECUTEQUERY",
    "EXECUTEQUERYTOLIST",
    "EXECUTESCROLLABLEQUERY",
    "EXECUTETRACKEDLISTQUERY",
    "EXPIRATIONBACKLOG",
    "EXPIRECACHE",
    "EXPIRED",
    "EXPIREDTABLESERIALS",
    "EXPLICITIDALIASREQUIREDINJOINS",
    "FIELDS",
    "FINALLYIMMUTABLE",
    "FINDDUPLICATE",
    "FORCEDMODIFIED",
    "FROMTHISJVM",
    "HASHCODE",
    "ID",
    "IDSOURCE",
    "IDVALID",
    "IMMUTABLE",
    "IMMUTABLELOGGINGLEVEL",
    "INSERTOBJECT",
    "INSERTPLAIN",
    "INSERTPLAINWITHCOMPONENTS",
    "LOADCOMPONENTS",
    "LOADLAZYREFERENCES",
    "LOGGINGMODIFICATION",
    "LOGMODIFICATION",
    "MARKCACHEACCESS",
    "MARKDELETED",
    "ME",
    "MODIFICATIONCOUNT",
    "MODIFICATIONLOG",
    "MODIFIED",
    "NEW",
    "NEWID",
    "NEWINSTANCE",
    "NORMTEXT",
    "NORMTEXTPROVIDED",
    "NOTIFY",
    "NOTIFYALL",
    "ON",
    "OP",
    "ORDERBY",
    "OVERLOADABLE",
    "PDO",
    "PDOCLASS",
    "PDOMETHODCACHE",
    "PERMISSIONACCEPTED",
    "PERSIST",
    "PERSISTABLE",
    "PERSISTENCECLASS",
    "PERSISTOBJECT",
    "PERSISTTOKENLOCKED",
    "PREPAREDELETE",
    "PREPAREDSTATEMENT",
    "PREPARESAVE",
    "PREPARESETFIELDS",
    "READALLOWED",
    "READFROMRESULTSETWRAPPER",
    "READJOINEDROW",
    "REFERENCED",
    "RELEASETOKENLOCK",
    "RELOAD",
    "RELOADFORUPDATE",
    "RELOADOBJECT",
    "RELOADOBJECTFORUPDATE",
    "REMOTEDELEGATE",
    "REMOVABLE",
    "REMOVEALLPROPERTYLISTENERS",
    "REMOVEPROPERTYLISTENER",
    "REMOVEREFERENCINGCLASS",
    "RENEWTOKENLOCKREQUESTED",
    "REPLAYEDLENIENTLY",
    "REQUESTTOKENLOCK",
    "RESERVEID",
    "RESULTALL",
    "RESULTALLCURSOR",
    "RESULTALLIDSERIAL",
    "RESULTALLOBJECTS",
    "RESULTALLWITHEXPIREDTABLESERIALS",
    "RESULTBYNORMTEXT",
    "RESULTBYNORMTEXTCURSOR",
    "RESULTOBJECTSWITHEXPIREDTABLESERIALS",
    "REVERTTOSNAPSHOT",
    "ROOTCLASSID",
    "ROOTCLASSIDPROVIDED",
    "ROOTENTITY",
    "ROOTENTITYOF",
    "ROOTID",
    "ROOTIDPROVIDED",
    "SAVE",
    "SAVEOBJECT",
    "SAVEREFERENCEDRELATIONS",
    "SAVEREFERENCINGRELATIONS",
    "SECURITYRESULT",
    "SELECT",
    "SELECTALL",
    "SELECTALLASCURSOR",
    "SELECTALLCACHED",
    "SELECTALLFORCACHE",
    "SELECTALLIDSERIAL",
    "SELECTALLOBJECTS",
    "SELECTALLWITHEXPIREDTABLESERIALS",
    "SELECTBYNORMTEXT",
    "SELECTBYNORMTEXTASCURSOR",
    "SELECTBYTEMPLATE",
    "SELECTCACHED",
    "SELECTCACHEDONLY",
    "SELECTEXPIREDTABLESERIALS",
    "SELECTFORCACHE",
    "SELECTFORUPDATE",
    "SELECTLATEST",
    "SELECTMAXID",
    "SELECTMAXTABLESERIAL",
    "SELECTNEXTOBJECT",
    "SELECTOBJECT",
    "SELECTOBJECTFORUPDATE",
    "SELECTOBJECTSWITHEXPIREDTABLESERIALS",
    "SELECTSERIAL",
    "SERIAL",
    "SESSION",
    "SESSIONHOLDER",
    "SESSIONIMMUTABLE",
    "SNAPSHOT",
    "SNAPSHOTS",
    "SQLCLASSIDCONDITION",
    "SQLCONTEXTCONDITION",
    "STATEMENTALWAYSPREPARED",
    "TABLEALIAS",
    "TABLENAME",
    "TABLESERIAL",
    "TABLESERIALPROVIDED",
    "TOGENERICSTRING",
    "TOIDSTRING",
    "TOKENLOCKABLEBYME",
    "TOKENLOCKED",
    "TOKENLOCKEDBY",
    "TOKENLOCKEDBYME",
    "TOKENLOCKPROVIDED",
    "TOKENLOCKTIMEOUT",
    "TOPSUPERTABLEALIAS",
    "TOPSUPERTABLENAME",
    "TOSTRING",
    "TRACKED",
    "TRANSFERTOKENLOCK",
    "TRANSIENTDATA",
    "UNMARKDELETED",
    "UPDATENORMTEXT",
    "UPDATEOBJECT",
    "UPDATEPLAIN",
    "UPDATEROOTCONTEXT",
    "UPDATESERIAL",
    "UPDATESERIALANDTABLESERIAL",
    "UPDATETOKENLOCK",
    "UPDATETOKENLOCKONLY",
    "UPDATINGSERIALEVENIFNOTMODIFIED",
    "VALIDATE",
    "VALIDATED",
    "VALIDCLASSIDS",
    "VIEWALLOWED",
    "VIRGIN",
    "WAIT",
    "WRITEALLOWED"
  );

  /**
   * Creates the name verifier.
   */
  public NameVerifierImpl() {
    // see -Xlint:missing-explicit-ctor since Java 16
  }

  @Override
  public String verifyEntityName(Entity entity) {
    String name = entity.getName();
    return StringHelper.isValidJavaClassName(name) ? null : "'" + name + "' is not a valid Java class name";
  }

  @Override
  public String verifyTableName(Entity entity) {
    String name = entity.getTableName();
    if (isNotQuotedOrProvided(name, entity)) {
      return isAllUpperOrLowerCase(name, entity) ? null : TABLE_NAME + name + MIXED_UC_LC;
    }
    return TABLE_NAME + name + QUOTED;
  }

  @Override
  public String verifyTableAlias(Entity entity) {
    String name = entity.getTableAlias();
    if (isNotQuotedOrProvided(name, entity)) {
      return isAllUpperOrLowerCase(name, entity) ? null : TABLE_ALIAS + name + MIXED_UC_LC;
    }
    return TABLE_ALIAS + name + QUOTED;
  }

  @Override
  public String verifyAttributeName(Attribute attribute) {
    return attribute.isImplicit() ? null : validateJavaIdentifier(attribute.getName());
  }

  @Override
  public String verifyRelationName(Relation relation) {
    return validateJavaIdentifier(relation.getVariableName());
  }

  @Override
  public String verifyColumnName(Attribute attribute) {
    String name = attribute.getColumnName();
    if (isNotQuotedOrProvided(name, attribute.getEntity())) {
      return isAllUpperOrLowerCase(name, attribute.getEntity()) ? null : COLUMN_NAME + name + MIXED_UC_LC;
    }
    return COLUMN_NAME + name + QUOTED;
  }

  @Override
  public String verifyIndexName(Index index) {
    String name = index.getName();  // index not allowed for [PROVIDED] entities
    return isAllUpperOrLowerCase(name, index.getEntity()) ? null : "index name '" + name + MIXED_UC_LC;
  }


  /**
   * Checks whether the given name is all upper- or all lowercase.<br>
   * The backends are usually not case-sensitive (unless the SQL names are quoted),
   * but it is good style to either use all upper- or all lowercase.
   * Mixing the case pretends that the case is important, but in fact it's not.
   * This is only allowed for provided entities.
   *
   * @param name the name
   * @param entity the entity
   * @return true if ok, false if mixed
   */
  protected boolean isAllUpperOrLowerCase(String name, Entity entity) {
    return name == null || entity.getOptions().isProvided() || name.equals(name.toLowerCase(Locale.ROOT)) || name.equals(name.toUpperCase(Locale.ROOT));
  }

  /**
   * Checks whether the name is quoted.
   *
   * @param name the name
   * @param entity the entity
   * @return true if not quoted or entity is provided by another application
   */
  protected boolean isNotQuotedOrProvided(String name, Entity entity) {
    return name == null || !name.startsWith("\"") || entity.getOptions().isProvided();
  }

  /**
   * Validates the given name against the rules for Java identifiers
   * and possible conflicts with method names in superclasses.
   *
   * @param name the attribute or relation name
   * @return null if ok, else error message
   */
  protected String validateJavaIdentifier(String name) {
    if (!StringHelper.isValidJavaIdentifier(name)) {
      return "'" + name + "' is not a valid Java identifier name";
    }
    if (getIllegalNames().contains(name)) {
      return "'" + name + "' conflicts with method in superclasses";
    }
    if (getConfusingNames().contains(name.toUpperCase(Locale.ROOT))) {
      return "'" + name + "' may be confused with similar method in superclasses";
    }
    return null;
  }

  /**
   * Gets the set of illegal names.
   *
   * @return illegal model names
   */
  protected Set<String> getIllegalNames() {
    return ILLEGAL_NAMES;
  }

  /**
   * Gets the set of confusing names.
   *
   * @return confusing model name
   */
  protected Set<String> getConfusingNames() {
    return CONFUSING_NAMES;
  }

}
