/**
 * Tentackle - http://www.tentackle.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.tentackle.swing;


/**
 * Radiobuttons to get a definitive decision from the user.<br>
 *
 * The user must click yes or no or the answer is undetermined.
 *
 * @author harald
 */
@SuppressWarnings("serial")
public class YesNoRadioButton extends FormComponentPanel {

  private Boolean savedValue;


  /**
   * Creates a YesNoRadioButton.
   */
  public YesNoRadioButton() {
    initComponents();
    setYesText(SwingSwingBundle.getString("YES"));
    setNoText(SwingSwingBundle.getString("NO"));
    setFormComponent(yesButton);
  }


  /**
   * Checks if user has chosen a value.
   *
   * @return true if neither yes nor no selected, false otherwise
   */
  public boolean isUndetermined() {
    return !yesButton.isSelected() && !noButton.isSelected();
  }


  /**
   * Gets the decision.
   *
   * @return Boolean.TRUE if yes, Boolean.FALSE if no, null if no decision by user
   */
  public Boolean getSelected() {
    return isUndetermined() ? null : yesButton.isSelected();
  }


  /**
   * @see #getSelected()
   */
  @Override
  public Boolean getFormValue() {
    return getSelected();
  }



  /**
   * Sets the value.
   *
   * @param selected is Boolean.TRUE if yes, Boolean.FALSE if no, null if no decision by user
   */
  public void setSelected(Boolean selected) {
    if (selected == null) {
      yesButton.setSelected(false);
      noButton.setSelected(false);
    }
    else  {
      yesButton.setSelected(selected);
      noButton.setSelected(!yesButton.isSelected());
    }
  }

  /**
   * @see #setSelected(Boolean)
   */
  @Override
  public void setFormValue(Object object) {
    setSelected(object instanceof Boolean ? (Boolean)object : null);
  }


  /**
   * Sets the text for the "yes" button.
   *
   * @param text the text for the "yes" button
   */
  public void setYesText(String text) {
    yesButton.setText(text);
  }

  /**
   * Gets the text for the "yes" button.
   *
   * @return the text for the "yes" button
   */
  public String getYesText() {
    return yesButton.getText();
  }

  /**
   * Sets the text for the "no" button.
   *
   * @param text the text for the "no" button
   */
  public void setNoText(String text) {
    noButton.setText(text);
  }

  /**
   * Gets the text for the "no" button.
   *
   * @return the text for the "no" button
   */
  public String getNoText() {
    return noButton.getText();
  }




  @Override
  public void setChangeable (boolean changeable)  {
    super.setChangeable(changeable);
    noButton.setChangeable(changeable);
  }

  @Override
  public void saveValue() {
    if (isHonourChangeable()) {
      savedValue = getSelected();
    }
  }

  @Override
  public boolean isValueChanged() {
    if (!isHonourChangeable()) {
      return false;
    }
    Boolean value = getSelected();
    if (value == null)  {
      return savedValue != null;
    }
    else  {
      return savedValue == null || savedValue != value.booleanValue();
    }
  }


  @Override
  public void setMandatory(boolean mandatory) {
    super.setMandatory(mandatory);
    noButton.setMandatory(mandatory);
  }



  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {
    java.awt.GridBagConstraints gridBagConstraints;

    yesButton = new org.tentackle.swing.FormRadioButton();
    noButton = new org.tentackle.swing.FormRadioButton();

    setLayout(new java.awt.GridBagLayout());

    yesButton.setText(SwingSwingBundle.getTranslation("YES")); // NOI18N
    yesButton.setName("yes"); // NOI18N
    yesButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        yesButtonActionPerformed(evt);
      }
    });
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 5);
    add(yesButton, gridBagConstraints);

    noButton.setText(SwingSwingBundle.getTranslation("NO")); // NOI18N
    noButton.setName("no"); // NOI18N
    noButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        noButtonActionPerformed(evt);
      }
    });
    add(noButton, new java.awt.GridBagConstraints());
  }// </editor-fold>//GEN-END:initComponents

  private void yesButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_yesButtonActionPerformed
    if (yesButton.isSelected()) {
      noButton.setSelected(false);
    }
    if (isAutoUpdate()) {
      fireValueEntered();
      FormUtilities.getInstance().doValidate(this);
    }
  }//GEN-LAST:event_yesButtonActionPerformed

  private void noButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_noButtonActionPerformed
    if (noButton.isSelected()) {
      yesButton.setSelected(false);
    }
    if (isAutoUpdate()) {
      fireValueEntered();
      FormUtilities.getInstance().doValidate(this);
    }
  }//GEN-LAST:event_noButtonActionPerformed


  // Variables declaration - do not modify//GEN-BEGIN:variables
  private org.tentackle.swing.FormRadioButton noButton;
  private org.tentackle.swing.FormRadioButton yesButton;
  // End of variables declaration//GEN-END:variables

}
