/**
 * Tentackle - http://www.tentackle.org
 * Copyright (C) 2001-2008 Harald Krake, harald@krake.de, +49 7722 9508-0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// Created on June 26, 2003, 11:28 AM

package org.tentackle.swing;

import java.awt.EventQueue;
import java.awt.event.WindowEvent;
import javax.swing.JProgressBar;



/**
 * A dialog showing the progress of an infinite worker thread.
 * <p>
 * Notice: the dialog can be used without a worker thread as well.
 *
 * @see WorkerThread
 */
@SuppressWarnings("serial")
public class WorkerDialog extends org.tentackle.swing.FormDialog {

  private final WorkerThread workerThread;        // the worker thread
  private boolean cancelAllowed;                  // true if closing the dialog is allowed


  /**
   * Creates a non-modal worker dialog.
   *
   * @param workerThread the worker thread to terminate if dialog is closed,
   *        null if none (and cancel will be disallowed)
   */
  public WorkerDialog(WorkerThread workerThread) {
    this.workerThread = workerThread;
    initComponents();
    setCancelAllowed(workerThread != null);
  }

  /**
   * Creates a worker dialog without a worker thread.
   */
  public WorkerDialog() {
    this(null);
  }


  /**
   * Sets whether cancel (or close) this dialog and terminating
   * the worker thread is allowed. If not allowed the cancel
   * button will not be visible and closing the dialog is inhibited.
   *
   * @param cancelAllowed true if allowed (default), false if not
   */
  public void setCancelAllowed(boolean cancelAllowed) {
    this.cancelAllowed = cancelAllowed;
    cancelButton.setVisible(cancelAllowed);
  }

  /**
   * Returns whether cancellling this dialog is allowed.
   *
   * @return true if cancel allowed (default)
   */
  public boolean isCancelAllowed() {
    return cancelAllowed;
  }

  /**
   * Gets the progress bar.
   * Useful to set the progress level of the bar.
   * Initially the progress bar in indeterminate.
   * @return the progress bar
   */
  public JProgressBar getProgressBar() {
    return progressBar;
  }


  /**
   * Sets the progress.<br>
   * If the progressbar is indeterminate it will be changed
   * to determinate and setStringPainted(true).
   * The update is performed by invokelater
   * because this method is usually called from the workerthread
   * (which is not the GUI thread).<br>
   *
   * @param n the progress value (by default 0...100)
   */
  public void setProgress(final int n) {
    EventQueue.invokeLater(() -> {
      if (progressBar.isIndeterminate()) {
        // initialize
        progressBar.setIndeterminate(false);
        progressBar.setStringPainted(true);
      }
      progressBar.setValue(n);
    });
  }



  @Override
  public void setTitle(String title) {
    super.setTitle(title);
    titleLabel.setText(title);
  }

  /**
   * {@inheritDoc}
   * <p>
   * Overridden to terminate the thread if window is closed
   */
  @SuppressWarnings("deprecation")
  @Override
  protected void processWindowEvent(WindowEvent e)  {
    if (e.getID() == WindowEvent.WINDOW_CLOSING) {
      if (!cancelAllowed || terminateWorkerThread()) {
        // cancel not allowed or
        // wait for workerthread to dispose this dialog...
        return;
      }
      // else: continue to dispose
    }
    super.processWindowEvent(e);
  }


  /**
   * Terminates the worker thread.
   *
   * @return true if workerthread got interrupt()ed, false if no running thread
   */
  private boolean terminateWorkerThread() {
    if (workerThread != null && workerThread.isAlive()) {
      // e.g. user aborted a printing dialog
      // interrupt the thread, hoping that it will catch that signal
      // properly...
      workerThread.interrupt();
      return true;
    }
    return false;
  }


  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {
    java.awt.GridBagConstraints gridBagConstraints;

    progressBar = new javax.swing.JProgressBar();
    cancelButton = new org.tentackle.swing.FormButton();
    titleLabel = new javax.swing.JLabel();

    setAutoPosition(true);
    getContentPane().setLayout(new java.awt.GridBagLayout());

    progressBar.setIndeterminate(true);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
    getContentPane().add(progressBar, gridBagConstraints);

    cancelButton.setIcon(org.tentackle.swing.plaf.PlafUtilities.getInstance().getIcon("cancel"));
    cancelButton.setText(SwingSwingBundle.getTranslation("CANCEL")); // NOI18N
    cancelButton.setName("cancel"); // NOI18N
    cancelButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        cancelButtonActionPerformed(evt);
      }
    });
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.insets = new java.awt.Insets(5, 0, 5, 5);
    getContentPane().add(cancelButton, gridBagConstraints);

    titleLabel.setFont(titleLabel.getFont().deriveFont(titleLabel.getFont().getStyle() | java.awt.Font.BOLD));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 5);
    getContentPane().add(titleLabel, gridBagConstraints);

    pack();
  }// </editor-fold>//GEN-END:initComponents

private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
  if (cancelAllowed && !terminateWorkerThread()) {
    // no running thread
    dispose();
  }
}//GEN-LAST:event_cancelButtonActionPerformed


  // Variables declaration - do not modify//GEN-BEGIN:variables
  private org.tentackle.swing.FormButton cancelButton;
  private javax.swing.JProgressBar progressBar;
  private javax.swing.JLabel titleLabel;
  // End of variables declaration//GEN-END:variables

}
