/**
 * Tentackle - http://www.tentackle.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.tentackle.swing;

import java.awt.Component;
import java.awt.GridBagConstraints;
import org.tentackle.swing.plaf.PlafUtilities;

/**
 *
 * @author harald
 */
@SuppressWarnings("serial")
public class FormSpinField extends FormComponentPanel implements ValueListener {


  /**
   * Creates a spin field.
   *
   * @param editingComponent the form component
   */
  public FormSpinField(FormComponent editingComponent) {
    initComponents();
    setFormComponent(editingComponent);
  }


  /**
   * Creates a spin field with a default editor {@link IntegerFormField}.
   */
  public FormSpinField () {
    this (new IntegerFormField());
  }


  /**
   * Gets the up button.
   *
   * @return the button
   */
  public FormButton getUpButton() {
    return upButton;
  }


  /**
   * Gets the down button.
   *
   * @return the button
   */
  public FormButton getDownButton() {
    return downButton;
  }


  @Override
  public void setFormComponent(FormComponent comp)  {
    if (getFormComponent() != null)  {
      removeValueListener(this);
      remove((Component)getFormComponent());
    }
    super.setFormComponent(comp);
    addValueListener(this);

    GridBagConstraints gridBagConstraints = new GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.gridheight = 2;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.weighty = 1.0;
    add((Component)comp, gridBagConstraints);
  }




  /**
   * Adds a SpinListener.
   *
   * @param listener the spin listener to add
   */
  public synchronized void addSpinListener (SpinListener listener) {
     listenerList.add (SpinListener.class, listener);
  }

  /**
   * Removes a SpinListener.
   *
   * @param listener the spin listener to remove
   */
  public synchronized void removeSpinListener (SpinListener listener) {
     listenerList.remove (SpinListener.class, listener);
  }


  /**
   * Notifies all SpinListeners that the field has
   * to be incremented.
   */
  public void fireIncrement () {
    Object listeners[] = listenerList.getListenerList();
    if (listeners != null)  {
      SpinEvent evt = null;
      for (int i = listeners.length-2; i >= 0; i -= 2)  {
        if (listeners[i] == SpinListener.class)  {
          if (evt == null) {
            evt = new SpinEvent(this, SpinEvent.INCREMENT);
          }
          ((SpinListener)listeners[i+1]).increment(evt, this);
        }
      }
    }
  }


  /**
   * Notifies all SpinListeners that the field has
   * to be decremented.
   */
  public void fireDecrement () {
    Object listeners[] = listenerList.getListenerList();
    if (listeners != null)  {
      SpinEvent evt = null;
      for (int i = listeners.length-2; i >= 0; i -= 2)  {
        if (listeners[i] == SpinListener.class)  {
          if (evt == null) {
            evt = new SpinEvent(this, SpinEvent.DECREMENT);
          }
          ((SpinListener)listeners[i+1]).decrement(evt, this);
        }
      }
    }
  }



  @Override
  public void valueChanged (ValueEvent e)  {
    if (isAutoUpdate()) {
      fireValueChanged();
    }
  }


  @Override
  public void valueEntered (ValueEvent e)  {
    if (isAutoUpdate()) {
      fireValueEntered();
      FormUtilities.getInstance().doValidate(this);
    }
  }


  /**
   * Spins one increment up.
   */
  public void spinUp() {
    fireIncrement();
    if (isAutoUpdate()) {
      fireValueEntered();
      FormUtilities.getInstance().doValidate(this);
    }
  }

  /**
   * Spins one increment down.
   */
  public void spinDown() {
    fireDecrement();
    if (isAutoUpdate()) {
      fireValueEntered();
      FormUtilities.getInstance().doValidate(this);
    }
  }


  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {
    java.awt.GridBagConstraints gridBagConstraints;

    upButton = new org.tentackle.swing.FormButton();
    downButton = new org.tentackle.swing.FormButton();

    setLayout(new java.awt.GridBagLayout());

    upButton.setIcon(PlafUtilities.getInstance().getIcon("spinup_mini"));
    upButton.setMargin(new java.awt.Insets(1, 1, 1, 1));
    upButton.setName("up"); // NOI18N
    upButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        upButtonActionPerformed(evt);
      }
    });
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
    gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 0);
    add(upButton, gridBagConstraints);

    downButton.setIcon(PlafUtilities.getInstance().getIcon("spindown_mini"));
    downButton.setMargin(new java.awt.Insets(1, 1, 1, 1));
    downButton.setName("down"); // NOI18N
    downButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        downButtonActionPerformed(evt);
      }
    });
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.SOUTH;
    gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 0);
    add(downButton, gridBagConstraints);
  }// </editor-fold>//GEN-END:initComponents

  private void upButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_upButtonActionPerformed
    spinUp();
  }//GEN-LAST:event_upButtonActionPerformed

  private void downButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_downButtonActionPerformed
    spinDown();
  }//GEN-LAST:event_downButtonActionPerformed


  // Variables declaration - do not modify//GEN-BEGIN:variables
  private org.tentackle.swing.FormButton downButton;
  private org.tentackle.swing.FormButton upButton;
  // End of variables declaration//GEN-END:variables

}
