/**
 * Tentackle - http://www.tentackle.org
 * Copyright (C) 2001-2008 Harald Krake, harald@krake.de, +49 7722 9508-0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.tentackle.swing;

import java.awt.Component;
import java.awt.Desktop;
import java.io.IOException;
import java.net.URI;
import org.tentackle.swing.plaf.PlafUtilities;


/**
 * FormField to edit a mail adress.<br>
 * By pressing the mail-button the mailer program
 * will be opened.
 *
 * @author harald
 */
@SuppressWarnings("serial")
public class FormMailField extends FormFieldComponentPanel {

  private String subject;                        // optional subject
  private String body;                           // optional body


  /**
   * Creates a form mail field.
   */
  public FormMailField() {
    initComponents();
    super.setFormComponent(mailField);
  }


  @Override
  public void setFormComponent(FormComponent comp) {
    throw new IllegalAccessError("changing the component is not allowed");
  }


  /**
   * {@inheritDoc}
   * <p>
   * Overridden to set the names in subcomponents.
   */
  @Override
  public void setName(String name) {
    super.setName(name);
    if (name != null) {
      ((Component) getFormComponent()).setName(name + "/mailAddress");
      mailButton.setName(name + "/sendMail");
    }
    else  {
      ((Component) getFormComponent()).setName("mailAddress");
      mailButton.setName("sendMail");
    }
  }


  /**
   * Gets the subject of the mail.
   *
   * @return the subject
   */
  public String getSubject() {
    return subject;
  }

  /**
   * Sets the subject of the mail that will
   * be passed to the mailer program when pressing the button.
   *
   * @param subject the subject
   */
  public void setSubject(String subject) {
    this.subject = subject;
  }

  /**
   * Gets the body of the mail.
   *
   * @return the body.
   */
  public String getBody() {
    return body;
  }

  /**
   * Sets the body of the mail that will
   * be passed to the mailer program when pressing the button.
   *
   * @param body the subject
   */
  public void setBody(String body) {
    this.body = body;
  }


  @Override
  public void setCellEditorUsage(boolean flag) {
    super.setCellEditorUsage(flag);
    /**
     * disable focus lost on datefield when used as a celleditor
     */
    mailButton.setFocusable(!flag);
  }


  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {
    java.awt.GridBagConstraints gridBagConstraints;

    mailField = new org.tentackle.swing.StringFormField();
    mailButton = new org.tentackle.swing.FormButton();

    setLayout(new java.awt.GridBagLayout());

    mailField.setName("mailAddress"); // NOI18N
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 1);
    add(mailField, gridBagConstraints);

    mailButton.setIcon(PlafUtilities.getInstance().getIcon("mail"));
    mailButton.setMargin(new java.awt.Insets(0, 0, 0, 0));
    mailButton.setName("sendMail"); // NOI18N
    mailButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        mailButtonActionPerformed(evt);
      }
    });
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.fill = java.awt.GridBagConstraints.VERTICAL;
    add(mailButton, gridBagConstraints);
  }// </editor-fold>//GEN-END:initComponents

  private void mailButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_mailButtonActionPerformed
    if (getText() != null && !getText().isEmpty())  {
      try {
        // start the mailer
        String url = getText();
        if (!url.toUpperCase().startsWith("MAILTO:")) {
          url = "mailto:" + url;
        }
        boolean withArgs = false;
        if (subject != null && !subject.isEmpty()) {
          url += "?subject=" + subject.replace(" ", "%20");
          withArgs = true;
        }
        if (body != null && !body.isEmpty()) {
          url += (withArgs?"&":"?") + "body=" + body.replace(" ", "%20").replace("\n", "%0d%0a");
        }
        Desktop.getDesktop().mail(URI.create(url));
      }
      catch (IOException ex) {
        FormError.showException(SwingSwingBundle.getString("MAILER COULD NOT BE LAUNCHED"), ex);
      }
    }
  }//GEN-LAST:event_mailButtonActionPerformed


  // Variables declaration - do not modify//GEN-BEGIN:variables
  private org.tentackle.swing.FormButton mailButton;
  private org.tentackle.swing.StringFormField mailField;
  // End of variables declaration//GEN-END:variables

}
