/*
 * Decompiled with CFR 0.152.
 */
package org.netcrusher.core.reactor;

import java.io.IOException;
import java.nio.channels.ClosedSelectorException;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.Iterator;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import org.netcrusher.NetCrusherException;
import org.netcrusher.core.nio.SelectionKeyCallback;
import org.netcrusher.core.reactor.NioSelectorPostOp;
import org.netcrusher.core.reactor.NioSelectorScheduledOp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NioSelector {
    private static final Logger LOGGER = LoggerFactory.getLogger(NioSelector.class);
    private static final long THREAD_TERMINATION_TIMEOUT_MS = 5000L;
    private final Thread thread;
    private final Selector selector;
    private final Queue<NioSelectorPostOp> postOperationQueue;
    private final Queue<NioSelectorScheduledOp> scheduledOperationQueue;
    private final long tickMs;
    private volatile boolean open;

    NioSelector(long tickMs) throws IOException {
        if (tickMs <= 0L) {
            throw new IllegalArgumentException("Tick period must be positive");
        }
        this.selector = Selector.open();
        this.postOperationQueue = new ConcurrentLinkedQueue<NioSelectorPostOp>();
        this.scheduledOperationQueue = new PriorityQueue<NioSelectorScheduledOp>();
        this.thread = new Thread(this::loop);
        this.thread.setName("NetCrusher selector event loop");
        this.thread.setDaemon(false);
        this.thread.start();
        this.tickMs = tickMs;
        this.open = true;
    }

    synchronized void close() {
        if (this.open) {
            int activeSelectionKeys;
            LOGGER.debug("Selector is closing");
            boolean interrupted = false;
            this.postOperationQueue.clear();
            this.wakeup();
            if (this.thread.isAlive()) {
                this.thread.interrupt();
                try {
                    this.thread.join(5000L);
                }
                catch (InterruptedException e) {
                    interrupted = true;
                }
                if (this.thread.isAlive()) {
                    LOGGER.error("NetCrusher selector thread is still alive");
                }
            }
            if ((activeSelectionKeys = this.selector.keys().size()) > 0) {
                LOGGER.warn("Selector still has {} selection keys. Have you closed all linked crushers before?", (Object)activeSelectionKeys);
            }
            try {
                this.selector.close();
            }
            catch (IOException e) {
                LOGGER.error("Fail to close selector", e);
            }
            this.open = false;
            LOGGER.debug("Selector is closed");
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
    }

    public SelectionKey register(SelectableChannel channel, int options, SelectionKeyCallback callback) {
        return this.execute(() -> channel.register(this.selector, options, callback));
    }

    public int wakeup() {
        return this.execute(this.selector::selectNow);
    }

    public <T> T execute(Callable<T> callable) throws NetCrusherException {
        if (this.open) {
            if (Thread.currentThread().equals(this.thread)) {
                try {
                    return callable.call();
                }
                catch (Exception e) {
                    throw new NetCrusherException("Fail to execute selector op", e);
                }
            }
            NioSelectorPostOp<T> postOperation = new NioSelectorPostOp<T>(callable);
            this.postOperationQueue.add(postOperation);
            this.selector.wakeup();
            try {
                return postOperation.await();
            }
            catch (InterruptedException e) {
                throw new NetCrusherException("Reactor operation was interrupted", e);
            }
            catch (ExecutionException e) {
                throw new NetCrusherException("Selector operation has failed", e);
            }
        }
        throw new IllegalStateException("Selector is closed");
    }

    public void schedule(Runnable runnable, long delayNs) {
        if (this.tickMs == 0L) {
            throw new IllegalStateException("Tick value should be set on selector");
        }
        if (!Thread.currentThread().equals(this.thread)) {
            throw new IllegalStateException("Scheduling only should be made fron selector's thread");
        }
        long nowNs = System.nanoTime();
        this.scheduledOperationQueue.add(new NioSelectorScheduledOp(nowNs + delayNs, runnable));
    }

    private void loop() {
        LOGGER.debug("Selector event loop started");
        while (!Thread.currentThread().isInterrupted()) {
            int count;
            try {
                count = this.selector.select(this.tickMs);
            }
            catch (ClosedSelectorException e) {
                break;
            }
            catch (Exception e) {
                LOGGER.error("Error on select()", e);
                break;
            }
            if (count > 0) {
                Set<SelectionKey> keys = this.selector.selectedKeys();
                Iterator<SelectionKey> keyIterator = keys.iterator();
                while (keyIterator.hasNext()) {
                    SelectionKey selectionKey = keyIterator.next();
                    if (selectionKey.isValid()) {
                        SelectionKeyCallback callback = (SelectionKeyCallback)selectionKey.attachment();
                        try {
                            callback.execute(selectionKey);
                        }
                        catch (Exception e) {
                            LOGGER.error("Error while executing selection key callback", e);
                        }
                    } else {
                        LOGGER.debug("Selection key is invalid: {}", (Object)selectionKey);
                    }
                    keyIterator.remove();
                }
            }
            this.runScheduledOperations();
            this.runPostOperations();
        }
        LOGGER.debug("Selector event loop has finished");
    }

    private void runScheduledOperations() {
        NioSelectorScheduledOp scheduledOperation;
        while ((scheduledOperation = this.scheduledOperationQueue.peek()) != null && scheduledOperation.isReady()) {
            scheduledOperation = this.scheduledOperationQueue.poll();
            if (scheduledOperation == null) continue;
            scheduledOperation.run();
        }
    }

    private void runPostOperations() {
        NioSelectorPostOp postOperation;
        while ((postOperation = this.postOperationQueue.poll()) != null) {
            postOperation.run();
        }
    }
}

