/*
 * Decompiled with CFR 0.152.
 */
package org.netcrusher.datagram;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.netcrusher.core.buffer.BufferOptions;
import org.netcrusher.core.meter.RateMeterImpl;
import org.netcrusher.core.meter.RateMeters;
import org.netcrusher.core.nio.NioUtils;
import org.netcrusher.core.nio.SelectionKeyControl;
import org.netcrusher.core.reactor.NioReactor;
import org.netcrusher.core.state.BitState;
import org.netcrusher.core.throttle.Throttler;
import org.netcrusher.datagram.DatagramCrusher;
import org.netcrusher.datagram.DatagramCrusherSocketOptions;
import org.netcrusher.datagram.DatagramFilters;
import org.netcrusher.datagram.DatagramOuter;
import org.netcrusher.datagram.DatagramQueue;
import org.netcrusher.datagram.DatagramUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DatagramInner {
    private static final Logger LOGGER = LoggerFactory.getLogger(DatagramInner.class);
    private static final int DEFAULT_OUTER_CAPACITY = 32;
    private final DatagramCrusher crusher;
    private final NioReactor reactor;
    private final DatagramCrusherSocketOptions socketOptions;
    private final DatagramFilters filters;
    private final Meters meters;
    private final InetSocketAddress bindAddress;
    private final InetSocketAddress connectAddress;
    private final InetSocketAddress bindBeforeConnectAddress;
    private final DatagramChannel channel;
    private final SelectionKeyControl selectionKeyControl;
    private final ByteBuffer bb;
    private final Map<InetSocketAddress, DatagramOuter> outers;
    private final DatagramQueue incoming;
    private final BufferOptions bufferOptions;
    private final State state;

    DatagramInner(DatagramCrusher crusher, NioReactor reactor, DatagramCrusherSocketOptions socketOptions, BufferOptions bufferOptions, DatagramFilters filters, InetSocketAddress bindAddress, InetSocketAddress connectAddress, InetSocketAddress bindBeforeConnectAddress) throws IOException {
        this.crusher = crusher;
        this.reactor = reactor;
        this.filters = filters;
        this.socketOptions = socketOptions;
        this.bindAddress = bindAddress;
        this.connectAddress = connectAddress;
        this.bindBeforeConnectAddress = bindBeforeConnectAddress;
        this.outers = new ConcurrentHashMap<InetSocketAddress, DatagramOuter>(32);
        this.incoming = new DatagramQueue(bufferOptions);
        this.bufferOptions = bufferOptions;
        this.meters = new Meters();
        this.channel = DatagramChannel.open(socketOptions.getProtocolFamily());
        socketOptions.setupSocketChannel(this.channel);
        this.channel.bind(bindAddress);
        this.channel.configureBlocking(false);
        bufferOptions.checkDatagramSocket(this.channel.socket());
        this.bb = NioUtils.allocaleByteBuffer(this.channel.socket().getReceiveBufferSize(), bufferOptions.isDirect());
        SelectionKey selectionKey = reactor.getSelector().register(this.channel, 0, this::callback);
        this.selectionKeyControl = new SelectionKeyControl(selectionKey);
        this.state = new State(State.FROZEN);
        LOGGER.debug("Inner on <{}> is started", (Object)bindAddress);
    }

    void close() {
        this.reactor.getSelector().execute(() -> {
            if (this.state.not(State.CLOSED)) {
                if (this.state.is(State.OPEN)) {
                    this.freeze();
                }
                if (!this.incoming.isEmpty()) {
                    LOGGER.warn("On closing inner has {} incoming datagrams", (Object)this.incoming.size());
                }
                NioUtils.close(this.channel);
                Iterator<DatagramOuter> outerIterator = this.outers.values().iterator();
                while (outerIterator.hasNext()) {
                    DatagramOuter outer = outerIterator.next();
                    outerIterator.remove();
                    outer.close();
                    this.crusher.notifyOuterDeleted(outer);
                }
                this.reactor.getSelector().wakeup();
                this.state.set(State.CLOSED);
                LOGGER.debug("Inner on <{}> is closed", (Object)this.bindAddress);
                return true;
            }
            return false;
        });
    }

    private void closeAll() {
        this.close();
        this.crusher.close();
    }

    void unfreeze() {
        this.reactor.getSelector().execute(() -> {
            if (this.state.is(State.FROZEN)) {
                if (this.incoming.isEmpty()) {
                    this.selectionKeyControl.setReadsOnly();
                } else {
                    this.selectionKeyControl.setAll();
                }
                for (DatagramOuter outer : this.outers.values()) {
                    outer.unfreeze();
                }
                this.state.set(State.OPEN);
                return true;
            }
            throw new IllegalStateException("Inner is not frozen on unfreeze");
        });
    }

    void freeze() {
        this.reactor.getSelector().execute(() -> {
            if (this.state.is(State.OPEN)) {
                if (this.selectionKeyControl.isValid()) {
                    this.selectionKeyControl.setNone();
                }
                for (DatagramOuter outer : this.outers.values()) {
                    outer.freeze();
                }
                this.state.set(State.FROZEN);
                return true;
            }
            throw new IllegalStateException("Inner is not open on freeze");
        });
    }

    boolean isFrozen() {
        return this.state.isAnyOf(State.FROZEN | State.CLOSED);
    }

    private void callback(SelectionKey selectionKey) throws IOException {
        if (selectionKey.isWritable()) {
            try {
                this.handleWritableEvent(false);
            }
            catch (ClosedChannelException e) {
                LOGGER.debug("Channel is closed on write");
                this.closeAll();
            }
            catch (Exception e) {
                LOGGER.error("Exception in inner on write", e);
                this.closeAll();
            }
        }
        if (selectionKey.isReadable()) {
            try {
                this.handleReadableEvent();
            }
            catch (ClosedChannelException e) {
                LOGGER.debug("Channel is closed on read");
                this.closeAll();
            }
            catch (Exception e) {
                LOGGER.error("Exception in inner on read", e);
                this.closeAll();
            }
        }
    }

    private void handleWritableEvent(boolean forced) throws IOException {
        DatagramQueue.BufferEntry entry;
        int count = 0;
        while (this.state.isWritable() && (entry = this.incoming.request()) != null) {
            int sent;
            boolean emptyDatagram;
            long delayNs = entry.getScheduledNs() - System.nanoTime();
            if (delayNs > 0L) {
                this.throttleSend(delayNs);
                this.incoming.retry(entry);
                break;
            }
            boolean bl = emptyDatagram = !entry.getBuffer().hasRemaining();
            if (emptyDatagram && (count > 0 || forced)) {
                this.incoming.retry(entry);
                break;
            }
            try {
                sent = this.channel.send(entry.getBuffer(), entry.getAddress());
            }
            catch (SocketException e) {
                DatagramUtils.rethrowSocketException(e);
                this.incoming.retry(entry);
                break;
            }
            if (!emptyDatagram && sent <= 0) break;
            if (entry.getBuffer().hasRemaining()) {
                LOGGER.warn("Datagram is split");
                this.incoming.retry(entry);
            } else {
                this.incoming.release(entry);
            }
            this.meters.sentBytes.update(sent);
            this.meters.sentPackets.increment();
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Send {} bytes to client <{}>", (Object)sent, (Object)entry.getAddress());
            }
            ++count;
        }
        if (this.incoming.isEmpty()) {
            this.selectionKeyControl.disableWrites();
        }
    }

    private void handleReadableEvent() throws IOException {
        while (this.state.isReadable()) {
            this.bb.clear();
            InetSocketAddress address = (InetSocketAddress)this.channel.receive(this.bb);
            if (address == null) break;
            this.bb.flip();
            int read = this.bb.remaining();
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Received {} bytes from inner <{}>", (Object)read, (Object)address);
            }
            this.meters.readBytes.update(read);
            this.meters.readPackets.increment();
            DatagramOuter outer = this.requestOuter(address);
            outer.enqueue(this.bb);
        }
    }

    private void suggestDeferredSent() {
        if (!this.incoming.isEmpty() && this.state.isWritable()) {
            this.selectionKeyControl.enableWrites();
        }
    }

    private void suggestImmediateSent() throws IOException {
        if (!this.incoming.isEmpty() && this.state.isWritable()) {
            this.handleWritableEvent(true);
        }
    }

    void enqueue(InetSocketAddress clientAddress, ByteBuffer bbToCopy) throws IOException {
        Throttler throttler = this.filters.getIncomingGlobalThrottler();
        long delayNs = throttler != null ? throttler.calculateDelayNs(bbToCopy) : Throttler.NO_DELAY_NS;
        this.incoming.add(clientAddress, bbToCopy, delayNs);
        this.suggestImmediateSent();
        this.suggestDeferredSent();
    }

    private void throttleSend(long delayNs) {
        if (this.state.is(State.OPEN) && !this.state.isSendThrottled()) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Inner sent is throttled on {}ns", (Object)delayNs);
            }
            this.state.setSendThrottled(true);
            if (this.selectionKeyControl.isValid()) {
                this.selectionKeyControl.disableWrites();
            }
            this.reactor.getSelector().schedule(this::unthrottleSend, delayNs);
        }
    }

    private void unthrottleSend() {
        if (this.state.is(State.OPEN) && this.state.isSendThrottled()) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Inner sent is unthrottled");
            }
            this.state.setSendThrottled(false);
            if (this.selectionKeyControl.isValid() && this.state.isWritable() && !this.incoming.isEmpty()) {
                this.selectionKeyControl.enableWrites();
            }
        }
    }

    private DatagramOuter requestOuter(InetSocketAddress address) throws IOException {
        DatagramOuter outer = this.outers.get(address);
        if (outer == null) {
            outer = new DatagramOuter(this, this.reactor, this.socketOptions, this.filters, this.bufferOptions, address, this.connectAddress, this.bindBeforeConnectAddress);
            outer.unfreeze();
            this.outers.put(address, outer);
            this.meters.clientTotalCount.incrementAndGet();
            this.crusher.notifyOuterCreated(outer);
        }
        return outer;
    }

    boolean closeOuter(InetSocketAddress clientAddress) {
        DatagramOuter outer = this.outers.remove(clientAddress);
        if (outer != null) {
            outer.close();
            this.crusher.notifyOuterDeleted(outer);
            return true;
        }
        return false;
    }

    int closeIdleOuters(long maxIdleDurationMs) {
        int countBefore = this.outers.size();
        if (countBefore > 0) {
            Iterator<DatagramOuter> outerIterator = this.outers.values().iterator();
            while (outerIterator.hasNext()) {
                DatagramOuter outer = outerIterator.next();
                if (outer.getIdleDurationMs() <= maxIdleDurationMs) continue;
                outerIterator.remove();
                outer.close();
                this.crusher.notifyOuterDeleted(outer);
            }
            int countAfter = this.outers.size();
            return countBefore - countAfter;
        }
        return 0;
    }

    DatagramOuter getOuter(InetSocketAddress clientAddress) {
        return this.outers.get(clientAddress);
    }

    Collection<DatagramOuter> getOuters() {
        return this.outers.values();
    }

    RateMeters getByteMeters() {
        return new RateMeters(this.meters.readBytes, this.meters.sentBytes);
    }

    RateMeters getPacketMeters() {
        return new RateMeters(this.meters.readPackets, this.meters.sentPackets);
    }

    int getClientTotalCount() {
        return this.meters.clientTotalCount.get();
    }

    private static final class Meters {
        private final RateMeterImpl sentBytes = new RateMeterImpl();
        private final RateMeterImpl readBytes = new RateMeterImpl();
        private final RateMeterImpl sentPackets = new RateMeterImpl();
        private final RateMeterImpl readPackets = new RateMeterImpl();
        private final AtomicInteger clientTotalCount = new AtomicInteger(0);

        private Meters() {
        }
    }

    private static final class State
    extends BitState {
        private static final int OPEN = State.bit(0);
        private static final int FROZEN = State.bit(1);
        private static final int CLOSED = State.bit(2);
        private boolean sendThrottled = false;

        private State(int state) {
            super(state);
        }

        private boolean isWritable() {
            return this.is(OPEN) && !this.sendThrottled;
        }

        private boolean isReadable() {
            return this.is(OPEN);
        }

        private boolean isSendThrottled() {
            return this.sendThrottled;
        }

        private void setSendThrottled(boolean sendThrottled) {
            this.sendThrottled = sendThrottled;
        }
    }
}

