/*
 * Decompiled with CFR 0.152.
 */
package org.netcrusher.datagram;

import java.io.EOFException;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.PortUnreachableException;
import java.net.SocketAddress;
import java.net.SocketException;
import java.net.SocketOption;
import java.net.StandardSocketOptions;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.UnresolvedAddressException;
import org.netcrusher.core.buffer.BufferOptions;
import org.netcrusher.core.filter.PassFilter;
import org.netcrusher.core.filter.TransformFilter;
import org.netcrusher.core.meter.RateMeterImpl;
import org.netcrusher.core.meter.RateMeters;
import org.netcrusher.core.nio.NioUtils;
import org.netcrusher.core.nio.SelectionKeyControl;
import org.netcrusher.core.reactor.NioReactor;
import org.netcrusher.core.state.BitState;
import org.netcrusher.core.throttle.Throttler;
import org.netcrusher.datagram.DatagramCrusherSocketOptions;
import org.netcrusher.datagram.DatagramFilters;
import org.netcrusher.datagram.DatagramInner;
import org.netcrusher.datagram.DatagramQueue;
import org.netcrusher.datagram.DatagramUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DatagramOuter {
    private static final Logger LOGGER = LoggerFactory.getLogger(DatagramOuter.class);
    private final DatagramInner inner;
    private final NioReactor reactor;
    private final InetSocketAddress clientAddress;
    private final InetSocketAddress connectAddress;
    private final DatagramQueue incoming;
    private final DatagramChannel channel;
    private final SelectionKeyControl selectionKeyControl;
    private final Meters meters;
    private final Filters filters;
    private final ByteBuffer bb;
    private final State state;
    private volatile long lastOperationTimestamp;

    DatagramOuter(DatagramInner inner, NioReactor reactor, DatagramCrusherSocketOptions socketOptions, DatagramFilters filters, BufferOptions bufferOptions, InetSocketAddress clientAddress, InetSocketAddress connectAddress, InetSocketAddress bindBeforeConnectAddress) throws IOException {
        this.inner = inner;
        this.reactor = reactor;
        this.clientAddress = clientAddress;
        this.connectAddress = connectAddress;
        this.incoming = new DatagramQueue(bufferOptions);
        this.lastOperationTimestamp = System.currentTimeMillis();
        this.meters = new Meters();
        this.filters = new Filters(filters, clientAddress);
        this.channel = DatagramChannel.open(socketOptions.getProtocolFamily());
        socketOptions.setupSocketChannel(this.channel);
        this.channel.configureBlocking(false);
        bufferOptions.checkDatagramSocket(this.channel.socket());
        if (bindBeforeConnectAddress != null) {
            this.channel.setOption((SocketOption)StandardSocketOptions.SO_REUSEADDR, (Object)true);
            this.channel.bind(bindBeforeConnectAddress);
        }
        this.bb = NioUtils.allocaleByteBuffer(this.channel.socket().getReceiveBufferSize(), bufferOptions.isDirect());
        SelectionKey selectionKey = reactor.getSelector().register(this.channel, 0, this::callback);
        this.selectionKeyControl = new SelectionKeyControl(selectionKey);
        this.state = new State(State.FROZEN);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Outer for <{}> to <{}> is started", (Object)clientAddress, (Object)connectAddress);
        }
    }

    void close() {
        this.reactor.getSelector().execute(() -> {
            if (this.state.not(State.CLOSED)) {
                if (this.state.is(State.OPEN)) {
                    this.freeze();
                }
                if (!this.incoming.isEmpty()) {
                    LOGGER.warn("On closing outer has {} incoming datagrams", (Object)this.incoming.size());
                }
                NioUtils.close(this.channel);
                this.state.set(State.CLOSED);
                LOGGER.debug("Outer for <{}> to <{}> is closed", (Object)this.clientAddress, (Object)this.connectAddress);
                return true;
            }
            return false;
        });
    }

    private void closeAll() {
        this.close();
        this.inner.closeOuter(this.clientAddress);
    }

    void unfreeze() {
        this.reactor.getSelector().execute(() -> {
            if (this.state.is(State.FROZEN)) {
                if (this.incoming.isEmpty()) {
                    this.selectionKeyControl.setReadsOnly();
                } else {
                    this.selectionKeyControl.setAll();
                }
                this.state.set(State.OPEN);
                return true;
            }
            return false;
        });
    }

    void freeze() {
        this.reactor.getSelector().execute(() -> {
            if (this.state.is(State.OPEN)) {
                if (this.selectionKeyControl.isValid()) {
                    this.selectionKeyControl.setNone();
                }
                this.state.set(State.FROZEN);
                return true;
            }
            return false;
        });
    }

    private void callback(SelectionKey selectionKey) throws IOException {
        if (selectionKey.isWritable()) {
            try {
                this.handleWritableEvent(false);
            }
            catch (ClosedChannelException e) {
                LOGGER.debug("Channel is closed on write");
                this.closeAll();
            }
            catch (PortUnreachableException e) {
                LOGGER.debug("Port <{}> is unreachable on write", (Object)this.connectAddress);
                this.closeAll();
            }
            catch (UnresolvedAddressException e) {
                LOGGER.error("Connect address <{}> is unresolved", (Object)this.connectAddress);
                this.closeAll();
            }
            catch (Exception e) {
                LOGGER.error("Exception in outer on write", e);
                this.closeAll();
            }
        }
        if (selectionKey.isReadable()) {
            try {
                this.handleReadableEvent();
            }
            catch (ClosedChannelException e) {
                LOGGER.debug("Channel is closed on read");
                this.closeAll();
            }
            catch (EOFException e) {
                LOGGER.debug("EOF on read");
                this.closeAll();
            }
            catch (PortUnreachableException e) {
                LOGGER.debug("Port <{}> is unreachable on read", (Object)this.connectAddress);
                this.closeAll();
            }
            catch (Exception e) {
                LOGGER.error("Exception in outer on read", e);
                this.closeAll();
            }
        }
    }

    private void handleWritableEvent(boolean forced) throws IOException {
        DatagramQueue.BufferEntry entry;
        int count = 0;
        while (this.state.isWritable() && (entry = this.incoming.request()) != null) {
            int sent;
            boolean emptyDatagram;
            long delayNs = entry.getScheduledNs() - System.nanoTime();
            if (delayNs > 0L) {
                this.throttleSend(delayNs);
                this.incoming.retry(entry);
                break;
            }
            boolean bl = emptyDatagram = !entry.getBuffer().hasRemaining();
            if (emptyDatagram && (count > 0 || forced)) {
                this.incoming.retry(entry);
                break;
            }
            try {
                sent = this.channel.send(entry.getBuffer(), entry.getAddress());
            }
            catch (SocketException e) {
                DatagramUtils.rethrowSocketException(e);
                this.incoming.retry(entry);
                break;
            }
            if (!emptyDatagram && sent <= 0) break;
            if (entry.getBuffer().hasRemaining()) {
                LOGGER.warn("Datagram is split");
                this.incoming.retry(entry);
            } else {
                this.incoming.release(entry);
            }
            this.meters.sentBytes.update(sent);
            this.meters.sentPackets.increment();
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Send {} bytes to client <{}>", (Object)sent, (Object)entry.getAddress());
            }
            ++count;
            this.lastOperationTimestamp = System.currentTimeMillis();
        }
        if (this.incoming.isEmpty()) {
            this.selectionKeyControl.disableWrites();
        }
    }

    private void handleReadableEvent() throws IOException {
        while (this.state.isReadable()) {
            this.bb.clear();
            SocketAddress address = this.channel.receive(this.bb);
            if (address == null) break;
            if (!this.connectAddress.equals(address)) {
                LOGGER.trace("Datagram from non-connect address <{}> will be dropped", (Object)address);
                continue;
            }
            this.bb.flip();
            int read = this.bb.remaining();
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Read {} bytes from outer for <{}>", (Object)read, (Object)this.clientAddress);
            }
            this.meters.readBytes.update(read);
            this.meters.readPackets.increment();
            boolean passed = this.filter(this.bb, this.filters.incomingPassFilter, this.filters.incomingTransferFilter);
            if (passed) {
                this.inner.enqueue(this.clientAddress, this.bb);
            }
            this.lastOperationTimestamp = System.currentTimeMillis();
        }
    }

    private void suggestDeferredSent() {
        if (!this.incoming.isEmpty() && this.state.isWritable()) {
            this.selectionKeyControl.enableWrites();
        }
    }

    private void suggestImmediateSent() throws IOException {
        if (!this.incoming.isEmpty() && this.state.isWritable()) {
            this.handleWritableEvent(true);
        }
    }

    void enqueue(ByteBuffer bbToCopy) throws IOException {
        boolean passed = this.filter(bbToCopy, this.filters.outgoingPassFilter, this.filters.outgoingTransferFilter);
        if (passed) {
            Throttler throttler = this.filters.outgoingThrottler;
            long delayNs = throttler != null ? throttler.calculateDelayNs(bbToCopy) : Throttler.NO_DELAY_NS;
            this.incoming.add(this.connectAddress, bbToCopy, delayNs);
            this.suggestImmediateSent();
            this.suggestDeferredSent();
        }
    }

    private boolean filter(ByteBuffer bbToCopy, PassFilter passFilter, TransformFilter transformFilter) {
        boolean passed;
        if (passFilter != null && !(passed = passFilter.check(bbToCopy))) {
            return false;
        }
        if (transformFilter != null) {
            transformFilter.transform(bbToCopy);
        }
        return true;
    }

    private void throttleSend(long delayNs) {
        if (this.state.is(State.OPEN) && !this.state.isSendThrottled()) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Outer sent is throttled on {}ns", (Object)delayNs);
            }
            this.state.setSendThrottled(true);
            if (this.selectionKeyControl.isValid()) {
                this.selectionKeyControl.disableWrites();
            }
            this.reactor.getSelector().schedule(this::unthrottleSend, delayNs);
        }
    }

    private void unthrottleSend() {
        if (this.state.is(State.OPEN) && this.state.isSendThrottled()) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Outer sent is unthrottled");
            }
            this.state.setSendThrottled(false);
            if (this.selectionKeyControl.isValid() && this.state.isWritable() && !this.incoming.isEmpty()) {
                this.selectionKeyControl.enableWrites();
            }
        }
    }

    InetSocketAddress getClientAddress() {
        return this.clientAddress;
    }

    long getIdleDurationMs() {
        return System.currentTimeMillis() - this.lastOperationTimestamp;
    }

    RateMeters getByteMeters() {
        return new RateMeters(this.meters.readBytes, this.meters.sentBytes);
    }

    RateMeters getPacketMeters() {
        return new RateMeters(this.meters.readPackets, this.meters.sentPackets);
    }

    private static final class Filters {
        private final TransformFilter outgoingTransferFilter;
        private final TransformFilter incomingTransferFilter;
        private final PassFilter outgoingPassFilter;
        private final PassFilter incomingPassFilter;
        private final Throttler outgoingThrottler;

        private Filters(DatagramFilters filters, InetSocketAddress clientAddress) {
            this.outgoingTransferFilter = filters.getOutgoingTransformFilterFactory() != null ? filters.getOutgoingTransformFilterFactory().allocate(clientAddress) : null;
            this.incomingTransferFilter = filters.getIncomingTransformFilterFactory() != null ? filters.getIncomingTransformFilterFactory().allocate(clientAddress) : null;
            this.outgoingPassFilter = filters.getOutgoingPassFilterFactory() != null ? filters.getOutgoingPassFilterFactory().allocate(clientAddress) : null;
            this.incomingPassFilter = filters.getIncomingPassFilterFactory() != null ? filters.getIncomingPassFilterFactory().allocate(clientAddress) : null;
            this.outgoingThrottler = filters.getOutgoingThrottlerFactory() != null ? filters.getOutgoingThrottlerFactory().allocate(clientAddress) : null;
        }
    }

    private static final class Meters {
        private final RateMeterImpl sentBytes;
        private final RateMeterImpl readBytes = new RateMeterImpl();
        private final RateMeterImpl sentPackets;
        private final RateMeterImpl readPackets;

        private Meters() {
            this.sentBytes = new RateMeterImpl();
            this.readPackets = new RateMeterImpl();
            this.sentPackets = new RateMeterImpl();
        }
    }

    private static final class State
    extends BitState {
        private static final int OPEN = State.bit(0);
        private static final int FROZEN = State.bit(1);
        private static final int CLOSED = State.bit(2);
        private boolean sendThrottled = false;

        private State(int state) {
            super(state);
        }

        private boolean isWritable() {
            return this.is(OPEN) && !this.sendThrottled;
        }

        private boolean isReadable() {
            return this.is(OPEN);
        }

        private boolean isSendThrottled() {
            return this.sendThrottled;
        }

        private void setSendThrottled(boolean sendThrottled) {
            this.sendThrottled = sendThrottled;
        }
    }
}

