/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.utilities.test.io;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.AclEntry;
import java.nio.file.attribute.AclEntryFlag;
import java.nio.file.attribute.AclEntryPermission;
import java.nio.file.attribute.AclFileAttributeView;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.nio.file.attribute.UserPrincipal;
import java.util.AbstractMap;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;
import org.terracotta.utilities.exec.Shell;
import org.terracotta.utilities.test.io.WindowsSpecialFolder;

public final class CommonFiles {
    private static final Logger LOGGER = LoggerFactory.getLogger(CommonFiles.class);
    private static final boolean IS_WINDOWS = System.getProperty("os.name").toLowerCase(Locale.ROOT).startsWith("win");
    private static final EnumMap<PosixFilePermission, PosixFilePermission> OWNER_TO_OTHER_MAPPING;

    @SuppressFBWarnings(value={"DMI_HARDCODED_ABSOLUTE_FILENAME"}, justification="'/var/tmp' is the FHS-designated name for *NIX OS")
    public static Path createCommonAppFile(Path path) throws IOException {
        Path normalizedPath = path.normalize();
        if (normalizedPath.getNameCount() == 0) {
            throw new IllegalArgumentException("Path \"" + path + "\" is an effectively empty path");
        }
        if (normalizedPath.isAbsolute()) {
            throw new IllegalArgumentException("Path \"" + path + "\" is not relative");
        }
        Path commonFolder = IS_WINDOWS ? WindowsSpecialFolder.COMMON_APPLICATION_DATA.get() : Paths.get("/var/tmp", new String[0]);
        if (!Files.exists(commonFolder, new LinkOption[0])) {
            throw new FileNotFoundException("Directory \"" + commonFolder + "\" does not exist");
        }
        Path commonFile = commonFolder.resolve(normalizedPath);
        Iterator<Path> pathIterator = normalizedPath.iterator();
        Path pathInProgress = commonFolder;
        while (pathIterator.hasNext()) {
            pathInProgress = pathInProgress.resolve(pathIterator.next());
            try {
                if (pathIterator.hasNext()) {
                    try {
                        Files.createDirectory(pathInProgress, new FileAttribute[0]);
                        LOGGER.info("Created \"{}\"", (Object)pathInProgress);
                    }
                    catch (FileAlreadyExistsException e) {
                        if (!Files.isDirectory(pathInProgress, new LinkOption[0])) {
                            LOGGER.error("Directory \"{}\" cannot be created; file/dead link already exists in its place", (Object)pathInProgress);
                            throw e;
                        }
                        if (Files.isSymbolicLink(pathInProgress)) {
                            Path originalPath = pathInProgress;
                            pathInProgress = pathInProgress.toRealPath(new LinkOption[0]);
                            LOGGER.debug("Path \"{}\" linked to \"{}\"", (Object)originalPath, (Object)pathInProgress);
                        }
                        LOGGER.info("Directory \"{}\" already exists; will attempt to update ACL/permissions", (Object)pathInProgress);
                    }
                } else {
                    try {
                        Files.createFile(pathInProgress, new FileAttribute[0]);
                        LOGGER.info("Created \"{}\"", (Object)pathInProgress);
                    }
                    catch (FileAlreadyExistsException e) {
                        LOGGER.info("File \"{}\" already exists; will attempt to update ACL/permissions", (Object)pathInProgress);
                    }
                }
            }
            catch (FileAlreadyExistsException e) {
                throw e;
            }
            catch (IOException e) {
                LOGGER.error("Unable to create \"{}\"", (Object)pathInProgress);
                throw e;
            }
            CommonFiles.copyOwnerPermissions(pathInProgress);
        }
        return commonFile;
    }

    private static void copyOwnerPermissions(Path path) {
        Set<String> fileAttributeViews = path.getFileSystem().supportedFileAttributeViews();
        if (fileAttributeViews.contains("posix")) {
            CommonFiles.updatePosixPermissions(path);
        } else if (fileAttributeViews.contains("acl")) {
            CommonFiles.updateAcl(path);
        } else {
            LOGGER.warn("Path \"{}\" supports neither ACL nor POSIX permissions ({}); permissions not updated", (Object)path, (Object)fileAttributeViews);
        }
    }

    private static void updateAcl(Path path) {
        AclEntry aclEntry;
        UserPrincipal allUsersPrincipal;
        AclFileAttributeView view;
        CommonFiles.logAcl("Before update", Level.DEBUG, path);
        List<AclEntry> aclEntryList = null;
        UserPrincipal owner = null;
        try {
            view = Files.getFileAttributeView(path, AclFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
            if (view != null) {
                aclEntryList = view.getAcl();
                owner = view.getOwner();
            }
        }
        catch (IOException e) {
            LOGGER.warn("Unable to get ACL for \"{}\"; ACL remains unchanged", (Object)path, (Object)e);
            return;
        }
        if (owner == null) {
            LOGGER.warn("Owner for \"{}\" is not identified; ; ACL remains unchanged", (Object)path);
            return;
        }
        String allUsersName = IS_WINDOWS ? WindowsWellKnownIdentities.AUTHENTICATED_USERS : "EVERYONE@";
        try {
            allUsersPrincipal = path.getFileSystem().getUserPrincipalLookupService().lookupPrincipalByName(allUsersName);
        }
        catch (IOException e) {
            LOGGER.warn("Unable to obtain principal representing \"{}\"; ACL remains unchanged", (Object)allUsersName, (Object)e);
            return;
        }
        UserPrincipal creatorOwnerPrincipal = null;
        if (IS_WINDOWS) {
            try {
                creatorOwnerPrincipal = path.getFileSystem().getUserPrincipalLookupService().lookupPrincipalByName(WindowsWellKnownIdentities.CREATOR_OWNER);
            }
            catch (IOException e) {
                LOGGER.warn("Unable to obtain principal representing \"{}\"; ignoring associated ACE", (Object)WindowsWellKnownIdentities.CREATOR_OWNER, (Object)e);
            }
        }
        boolean multipleAce = false;
        AclEntry existingOwnerEntry = null;
        AclEntry existingCreatorOwnerEntry = null;
        for (AclEntry entry : aclEntryList) {
            if (entry.principal().equals(owner)) {
                multipleAce |= existingOwnerEntry != null;
                existingOwnerEntry = entry;
                continue;
            }
            if (!entry.principal().equals(creatorOwnerPrincipal)) continue;
            multipleAce |= existingCreatorOwnerEntry != null;
            existingCreatorOwnerEntry = entry;
        }
        if (multipleAce) {
            LOGGER.warn("The ACL for \"{}\" contains multiple ACE for {}; abandoning ACL update", (Object)path, (Object)(owner + (creatorOwnerPrincipal == null ? "" : " or " + creatorOwnerPrincipal)));
            CommonFiles.logAcl("Duplicate ACE", Level.WARN, path);
            return;
        }
        if (existingOwnerEntry == null) {
            LOGGER.warn("Owner of \"{}\" - \"{}\" - has no ACL; ACL remains unchanged", (Object)path, (Object)owner);
            return;
        }
        AclEntry.Builder ownerBuilder = AclEntry.newBuilder(existingOwnerEntry);
        AclEntry.Builder allUsersBuilder = AclEntry.newBuilder(existingOwnerEntry).setPrincipal(allUsersPrincipal);
        if (existingCreatorOwnerEntry != null) {
            LinkedHashSet<AclEntryPermission> mergedPermissions = new LinkedHashSet<AclEntryPermission>(existingOwnerEntry.permissions());
            mergedPermissions.addAll(existingCreatorOwnerEntry.permissions());
            allUsersBuilder.setPermissions(mergedPermissions);
            ownerBuilder.setPermissions(mergedPermissions);
            LinkedHashSet<AclEntryFlag> mergedFlags = new LinkedHashSet<AclEntryFlag>(existingOwnerEntry.flags());
            mergedFlags.addAll(existingCreatorOwnerEntry.flags());
            mergedFlags.remove((Object)AclEntryFlag.INHERIT_ONLY);
            allUsersBuilder.setFlags(mergedFlags);
            ownerBuilder.setFlags(mergedFlags);
        }
        AclEntry allUsersEntry = allUsersBuilder.build();
        AclEntry replacementOwnerEntry = ownerBuilder.build();
        boolean allUsersEncountered = false;
        boolean foundOwner = false;
        ListIterator<AclEntry> entryIterator = aclEntryList.listIterator();
        while (entryIterator.hasNext()) {
            aclEntry = entryIterator.next();
            if (aclEntry.principal().equals(owner)) {
                entryIterator.set(replacementOwnerEntry);
                entryIterator.add(allUsersEntry);
                foundOwner = true;
                continue;
            }
            if (aclEntry.principal().equals(creatorOwnerPrincipal)) {
                entryIterator.remove();
                continue;
            }
            if (!aclEntry.principal().equals(allUsersPrincipal)) continue;
            if (foundOwner) {
                entryIterator.remove();
                continue;
            }
            allUsersEncountered = true;
        }
        if (allUsersEncountered) {
            entryIterator = aclEntryList.listIterator();
            while (entryIterator.hasNext() && !(aclEntry = entryIterator.next()).principal().equals(owner)) {
                if (!aclEntry.principal().equals(allUsersPrincipal)) continue;
                entryIterator.remove();
            }
        }
        try {
            LOGGER.info("Updating ACL on \"{}\"", (Object)path);
            view.setAcl(aclEntryList);
            CommonFiles.logAcl("After update", Level.DEBUG, path);
        }
        catch (IOException e) {
            LOGGER.error("Unable to alter ACL for \"{}\"; permissions remain unchanged", (Object)path, (Object)e);
        }
    }

    private static void updatePosixPermissions(Path path) {
        CommonFiles.logPermissions("Before update", Level.DEBUG, path);
        PosixFileAttributeView view = null;
        Set<PosixFilePermission> permissions = null;
        try {
            view = Files.getFileAttributeView(path, PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
            if (view != null) {
                PosixFileAttributes attributes = view.readAttributes();
                permissions = attributes.permissions();
            }
        }
        catch (IOException e) {
            LOGGER.error("Unable to get permissions for \"{}\"", (Object)path, (Object)e);
        }
        if (permissions != null) {
            List otherPermissions = permissions.stream().map(OWNER_TO_OTHER_MAPPING::get).filter(Objects::nonNull).collect(Collectors.toList());
            if (permissions.addAll(otherPermissions)) {
                try {
                    LOGGER.info("Updating permissions on \"{}\"", (Object)path);
                    view.setPermissions(permissions);
                    CommonFiles.logPermissions("After update", Level.DEBUG, path);
                }
                catch (IOException e) {
                    LOGGER.error("Unable to alter permissions for \"{}\"", (Object)path, (Object)e);
                }
            }
        }
    }

    private static void logPermissions(String description, Level logLevel, Path path) {
        LoggerBridge bridge = LoggerBridge.getInstance(LOGGER, logLevel);
        if (bridge.isLevelEnabled()) {
            try {
                PosixFileAttributeView view = Files.getFileAttributeView(path, PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
                if (view != null) {
                    PosixFileAttributes attributes = view.readAttributes();
                    Set<PosixFilePermission> permissions = attributes.permissions();
                    bridge.log("{}: POSIX permissions for \"{}\" owned by {}:\n    [{}] {}", description, path, attributes.owner(), PosixFilePermissions.toString(permissions), permissions.stream().map(Enum::name).collect(Collectors.joining("+")));
                } else {
                    bridge.log("POSIX permissions for \"{}\" not supported", path);
                }
            }
            catch (IOException e) {
                bridge.log("Unable to get POSIX permissions for \"{}\"", path, e);
            }
        }
    }

    private static void logAcl(String description, Level logLevel, Path path) {
        LoggerBridge bridge = LoggerBridge.getInstance(LOGGER, logLevel);
        if (bridge.isLevelEnabled()) {
            try {
                AclFileAttributeView view = Files.getFileAttributeView(path, AclFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
                if (view != null) {
                    bridge.log("{}: ACL for \"{}\" owned by {}:\n    {}", description, path, view.getOwner(), view.getAcl().stream().map(Object::toString).collect(Collectors.joining("\n    ")));
                } else {
                    bridge.log("ACL for \"{}\" not supported", path);
                }
            }
            catch (IOException e) {
                bridge.log("Unable to get ACL for \"{}\"", path, e);
            }
            if (IS_WINDOWS) {
                try {
                    Shell.Result result = Shell.execute(Shell.Encoding.CHARSET, "icacls", "\"" + path + "\"");
                    if (result.exitCode() == 0) {
                        bridge.log("ICACLS \"{}\"\n    {}", path, String.join((CharSequence)"\n    ", result.lines()));
                    } else {
                        bridge.log("Failed to run ICACLS for \"{}\"\n    {}", path, String.join((CharSequence)"\n    ", result.lines()));
                    }
                }
                catch (IOException e) {
                    bridge.log("Unable to run ICACLS for \"{}\"", path, e);
                }
            }
        }
    }

    static {
        EnumMap<PosixFilePermission, PosixFilePermission> map = new EnumMap<PosixFilePermission, PosixFilePermission>(PosixFilePermission.class);
        map.put(PosixFilePermission.OWNER_READ, PosixFilePermission.OTHERS_READ);
        map.put(PosixFilePermission.OWNER_WRITE, PosixFilePermission.OTHERS_WRITE);
        map.put(PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.OTHERS_EXECUTE);
        OWNER_TO_OTHER_MAPPING = map;
    }

    private static final class WindowsWellKnownIdentities {
        static final String AUTHENTICATED_USERS = WindowsWellKnownIdentities.convertStringSidToPrincipalName("S-1-5-11", "NT AUTHORITY\\Authenticated Users");
        static final String CREATOR_OWNER = WindowsWellKnownIdentities.convertStringSidToPrincipalName("S-1-3-0", "\\CREATOR OWNER");

        private WindowsWellKnownIdentities() {
        }

        private static String convertStringSidToPrincipalName(String stringSid, String defaultPrincipalName) {
            Shell.Result result;
            Object[] command = new String[]{"powershell.exe", "-NoLogo", "-NoProfile", "-NonInteractive", "-Command", "&{$ErrorActionPreference = 'Stop'; (New-Object System.Security.Principal.SecurityIdentifier('" + stringSid + "')).Translate([System.Security.Principal.NTAccount]).Value}"};
            try {
                result = Shell.execute(Shell.Encoding.CHARSET, (String[])command);
            }
            catch (IOException e) {
                LOGGER.error("{} failed; using \"{}\"", Arrays.toString(command), defaultPrincipalName, e);
                return defaultPrincipalName;
            }
            if (result.exitCode() == 0) {
                return result.lines().get(0);
            }
            LOGGER.warn("Unable to obtain user/group name for security identifier {}; using \"{}\"\n    {}", stringSid, defaultPrincipalName, String.join((CharSequence)"\n    ", result.lines()));
            return defaultPrincipalName;
        }
    }

    private static final class LoggerBridge {
        private static final Map<Map.Entry<Logger, Level>, LoggerBridge> INSTANCES = new HashMap<Map.Entry<Logger, Level>, LoggerBridge>();
        private final Logger delegate;
        private final Level level;
        private final MethodHandle isLevelEnabled;
        private final MethodHandle log;

        public static LoggerBridge getInstance(Logger delegate, Level level) {
            return INSTANCES.computeIfAbsent(new AbstractMap.SimpleImmutableEntry<Logger, Level>(delegate, level), e -> new LoggerBridge((Logger)e.getKey(), (Level)((Object)((Object)e.getValue()))));
        }

        private LoggerBridge(Logger delegate, Level level) {
            MethodHandle log;
            MethodHandle isLevelEnabled;
            this.delegate = Objects.requireNonNull(delegate, "delegate");
            this.level = Objects.requireNonNull(level, "level");
            String levelName = level.name().toLowerCase(Locale.ROOT);
            MethodHandles.Lookup lookup = MethodHandles.publicLookup();
            MethodType type = MethodType.methodType(Boolean.TYPE);
            try {
                isLevelEnabled = lookup.findVirtual(Logger.class, "is" + levelName.substring(0, 1).toUpperCase(Locale.ROOT) + levelName.substring(1) + "Enabled", type);
            }
            catch (IllegalAccessException | NoSuchMethodException e) {
                isLevelEnabled = null;
                delegate.error("Unable to resolve '{} {}({})' method on {}; will log at INFO level", type.returnType(), levelName, type.parameterList(), Logger.class, e);
            }
            this.isLevelEnabled = isLevelEnabled;
            type = MethodType.methodType(Void.TYPE, String.class, Object[].class);
            try {
                log = lookup.findVirtual(Logger.class, levelName, type);
            }
            catch (IllegalAccessException | NoSuchMethodException e) {
                log = null;
                delegate.error("Unable to resolve '{} {}({})' method on {}; will log at INFO level", type.returnType(), levelName, type.parameterList(), Logger.class, e);
            }
            this.log = log;
        }

        public boolean isLevelEnabled() {
            if (this.isLevelEnabled != null) {
                try {
                    return this.isLevelEnabled.invokeExact(this.delegate);
                }
                catch (Throwable throwable) {
                    this.delegate.error("Failed to call {}; presuming {} is enabled", new Object[]{this.isLevelEnabled, this.level, throwable});
                    return true;
                }
            }
            return this.delegate.isInfoEnabled();
        }

        public void log(String format, Object ... arguments) {
            if (this.log != null) {
                try {
                    this.log.invokeExact(this.delegate, format, arguments);
                }
                catch (Throwable throwable) {
                    this.delegate.error("Failed to call {}; logging at INFO level", (Object)this.log, (Object)throwable);
                    this.delegate.info(format, arguments);
                }
            } else {
                this.delegate.info(format, arguments);
            }
        }
    }
}

