/*
 * Copyright Terracotta, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.terracotta.management.capabilities.context;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;

/**
 * @author Ludovic Orban
 */
public final class CapabilityContext implements Serializable {

  private final Collection<Attribute> attributes;

  public CapabilityContext(Attribute... attributes) {
    this(Arrays.asList(attributes));
  }

  public CapabilityContext(Collection<Attribute> attributes) {
    this.attributes = attributes;
  }

  public Collection<Attribute> getAttributes() {
    return attributes;
  }

  public Collection<String> getRequiredAttributeNames() {
    Collection<String> names = new HashSet<String>();
    for (Attribute attribute : this.attributes) {
      if (attribute.isRequired()) {
        names.add(attribute.getName());
      }
    }
    return names;
  }

  public Collection<Attribute> getRequiredAttributes() {
    Collection<Attribute> attributes = new ArrayList<Attribute>(this.attributes.size());
    for (Attribute attribute : this.attributes) {
      if (attribute.isRequired()) {
        attributes.add(attribute);
      }
    }
    return attributes;
  }

  public static class Attribute {
    private final String name;
    private final boolean required;

    public Attribute(String name, boolean required) {
      this.name = name;
      this.required = required;
    }

    public String getName() {
      return name;
    }

    public boolean isRequired() {
      return required;
    }
  }

}
