/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.utilities.test.net;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.utilities.exec.Shell;
import org.terracotta.utilities.test.runtime.Os;

public class EphemeralPorts {
    private static final Logger LOGGER = LoggerFactory.getLogger(EphemeralPorts.class);
    private static final Range IANA_DEFAULT_RANGE = new Range(49152, 65535);
    private static Range range = null;

    public static synchronized Range getRange() {
        if (range == null) {
            range = EphemeralPorts.findRange();
        }
        return range;
    }

    private static Range findRange() {
        if (Os.isLinux()) {
            return new Linux().getRange();
        }
        if (Os.isMac()) {
            return new Mac().getRange();
        }
        if (Os.isWindows()) {
            return new Windows().getRange();
        }
        if (Os.isSolaris()) {
            return new SolarisAndHPUX(false).getRange();
        }
        if (Os.isAix()) {
            return new Aix().getRange();
        }
        if (Os.isHpux()) {
            return new SolarisAndHPUX(true).getRange();
        }
        throw new AssertionError((Object)("No support for this OS: " + Os.getOsName()));
    }

    private static String firstLine(String ... cmd) throws IOException {
        Shell.Result result = Shell.execute((Charset)Shell.Encoding.CHARSET, (String[])cmd);
        if (result.exitCode() == 0) {
            return (String)result.lines().get(0);
        }
        throw new IOException(String.format("Command %s failed; rc=%d:%n    %s", Arrays.toString(cmd), result.exitCode(), String.join((CharSequence)"\n    ", (Iterable<? extends CharSequence>)result)));
    }

    private static Iterable<String> allLines(String ... cmd) throws IOException {
        Shell.Result result = Shell.execute((Charset)Shell.Encoding.CHARSET, (String[])cmd);
        if (result.exitCode() == 0) {
            return result;
        }
        throw new IOException(String.format("Command %s failed; rc=%d:%n    %s", Arrays.toString(cmd), result.exitCode(), String.join((CharSequence)"\n    ", (Iterable<? extends CharSequence>)result)));
    }

    private static Properties asProperties(String ... cmd) throws IOException {
        Shell.Result result = Shell.execute((Charset)Shell.Encoding.CHARSET, (String[])cmd);
        if (result.exitCode() == 0) {
            Properties props = new Properties();
            try {
                props.load(new StringReader(String.join((CharSequence)System.lineSeparator(), (Iterable<? extends CharSequence>)result)));
            }
            catch (IOException e) {
                throw new IOException(String.format("Failed to load Properties from %s:%n    %s", Arrays.toString(cmd), String.join((CharSequence)"\n    ", (Iterable<? extends CharSequence>)result)), e);
            }
            return props;
        }
        throw new IOException(String.format("Command %s failed; rc=%d:%n    %s", Arrays.toString(cmd), result.exitCode(), String.join((CharSequence)"\n    ", (Iterable<? extends CharSequence>)result)));
    }

    public static void main(String[] args) {
        System.err.println(EphemeralPorts.getRange());
    }

    private static class Aix
    implements RangeGetter {
        private Aix() {
        }

        @Override
        public Range getRange() {
            try {
                Properties props = EphemeralPorts.asProperties(new String[]{"/usr/sbin/no", "-a"});
                int low = Integer.parseInt(props.getProperty("tcp_ephemeral_low"));
                int high = Integer.parseInt(props.getProperty("tcp_ephemeral_high"));
                return new Range(low, high);
            }
            catch (Exception e) {
                LOGGER.warn("Unable to determine tcp_ephemeral_{low,high}; using default IANA range", (Throwable)e);
                return IANA_DEFAULT_RANGE;
            }
        }
    }

    private static class Linux
    implements RangeGetter {
        private static final String source = "/proc/sys/net/ipv4/ip_local_port_range";

        private Linux() {
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        @SuppressFBWarnings(value={"DMI_HARDCODED_ABSOLUTE_FILENAME"})
        public Range getRange() {
            int DEFAULT_LOWER = 32768;
            int DEFAULT_UPPER = 61000;
            Range defaultRange = new Range(32768, 61000);
            File src = new File(source);
            if (!src.exists() || !src.canRead()) {
                LOGGER.warn("Cannot access \"{}\"; using default range of [{}]", (Object)source, (Object)defaultRange);
                return defaultRange;
            }
            try (BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(src), StandardCharsets.UTF_8));){
                String data = reader.readLine();
                if (data == null) {
                    LOGGER.warn("Unexpected EOF on \"{}\"; using default range of [{}]", (Object)source, (Object)defaultRange);
                    Range range = defaultRange;
                    return range;
                }
                String[] parts = data.split("[ \\t]");
                if (parts.length != 2) {
                    LOGGER.warn("Unexpected number of tokens in \"{}\" - \"{}\"; using default range of [{}]", new Object[]{source, data, defaultRange});
                    Range range = defaultRange;
                    return range;
                }
                int low = Integer.parseInt(parts[0]);
                int high = Integer.parseInt(parts[1]);
                Range range = new Range(low, high);
                return range;
            }
            catch (IOException ioe) {
                LOGGER.warn("Unable to determine ipv4/ip_local_port_range; using default range of [{}]", (Object)defaultRange, (Object)ioe);
                return defaultRange;
            }
        }
    }

    private static class Mac
    implements RangeGetter {
        private Mac() {
        }

        @Override
        public Range getRange() {
            try {
                Properties props = EphemeralPorts.asProperties(new String[]{"sysctl", "net.inet.ip.portrange"});
                int low = Integer.parseInt(props.getProperty("net.inet.ip.portrange.hifirst"));
                int high = Integer.parseInt(props.getProperty("net.inet.ip.portrange.hilast"));
                return new Range(low, high);
            }
            catch (Exception e) {
                LOGGER.warn("Unable to determine net.inet.ip.portrange.{hifirst,hilast}; using default IANA range", (Throwable)e);
                return IANA_DEFAULT_RANGE;
            }
        }
    }

    private static class Windows
    implements RangeGetter {
        private Windows() {
        }

        @Override
        public Range getRange() {
            String osName = System.getProperty("os.name");
            if (osName.equalsIgnoreCase("windows xp") || osName.equalsIgnoreCase("windows 2003")) {
                return this.getLegacySettings();
            }
            return this.getNetshRange();
        }

        private Range getNetshRange() {
            try {
                File netshExe = new File(Os.findWindowsSystemRoot(), "netsh.exe");
                int start = -1;
                int num = -1;
                Pattern pattern = Pattern.compile("^.*: (\\p{XDigit}+)");
                for (String line : EphemeralPorts.allLines(new String[]{netshExe.getAbsolutePath(), "int", "ipv4", "show", "dynamicport", "tcp"})) {
                    Matcher matcher = pattern.matcher(line);
                    if (start == -1 && matcher.matches()) {
                        start = Integer.parseInt(matcher.group(1));
                        continue;
                    }
                    if (num == -1 && matcher.matches()) {
                        num = Integer.parseInt(matcher.group(1));
                        continue;
                    }
                    if (start == -1 || num == -1) continue;
                    break;
                }
                if (num == -1 || start == -1) {
                    throw new Exception("start: " + start + ", num = " + num);
                }
                return new Range(start, start + num - 1);
            }
            catch (Exception e) {
                LOGGER.warn("Unable to determine IPv4 TCP dynamicport range; using default IANA range", (Throwable)e);
                return IANA_DEFAULT_RANGE;
            }
        }

        private Range getLegacySettings() {
            int DEFAULT_LOWER = 1024;
            int DEFAULT_UPPER = 5000;
            Range defaultRange = new Range(1024, 5000);
            try {
                File regExe;
                String sysRoot = Os.findWindowsSystemRoot();
                if (sysRoot != null && (regExe = new File(sysRoot, "reg.exe")).exists()) {
                    Pattern pattern = Pattern.compile("^.*MaxUserPort\\s+REG_DWORD\\s+0x(\\p{XDigit}+)");
                    for (String line : EphemeralPorts.allLines(new String[]{regExe.getAbsolutePath(), "query", "HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Services\\Tcpip\\Parameters", "/v", "MaxUserPort"})) {
                        Matcher matcher = pattern.matcher(line);
                        if (!matcher.matches()) continue;
                        int val = Integer.parseInt(matcher.group(1), 16);
                        return new Range(1024, val);
                    }
                }
                LOGGER.warn("Unable to locate reg.exe; using default range of [{}]", (Object)defaultRange);
            }
            catch (Exception e) {
                LOGGER.warn("Unable to determine TCP/IP MaxUserPort; using default range of [{}]", (Object)defaultRange, (Object)e);
            }
            return defaultRange;
        }
    }

    private static class SolarisAndHPUX
    implements RangeGetter {
        private final String ndd;

        public SolarisAndHPUX(boolean isHpux) {
            this.ndd = isHpux ? "/usr/bin/ndd" : "/usr/sbin/ndd";
        }

        @Override
        public Range getRange() {
            try {
                String lower = EphemeralPorts.firstLine(new String[]{this.ndd, "/dev/tcp", "tcp_smallest_anon_port"});
                String upper = EphemeralPorts.firstLine(new String[]{this.ndd, "/dev/tcp", "tcp_largest_anon_port"});
                int low = Integer.parseInt(lower.replaceAll("\n", ""));
                int high = Integer.parseInt(upper.replaceAll("\n", ""));
                return new Range(low, high);
            }
            catch (Exception e) {
                LOGGER.warn("Unable to determine tcp_{smallest,largest}_anon_port; using default IANA range", (Throwable)e);
                return IANA_DEFAULT_RANGE;
            }
        }
    }

    private static interface RangeGetter {
        public Range getRange();
    }

    public static class Range {
        private final int upper;
        private final int lower;

        private Range(int lower, int upper) {
            this.lower = lower;
            this.upper = upper;
        }

        public int getUpper() {
            return this.upper;
        }

        public int getLower() {
            return this.lower;
        }

        public boolean isInRange(int num) {
            return num >= this.lower && num <= this.upper;
        }

        public String toString() {
            return this.lower + " " + this.upper;
        }
    }
}

