/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.utilities.test.net;

import java.io.IOException;
import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.net.BindException;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.IntUnaryOperator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.utilities.test.net.EphemeralPorts;
import org.terracotta.utilities.test.net.SystemLevelLocker;

public class PortManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(PortManager.class);
    private static final PortManager INSTANCE = new PortManager();
    private static final int MAXIMUM_PORT_NUMBER = 65535;
    private static final int MINIMUM_ASSIGNABLE_PORT_COUNT = 1024;
    private static final InetAddress LOCALHOST;
    private final Random rnd = new SecureRandom();
    private final BitSet reservedPorts = new BitSet();
    private final BitSet portMap = new BitSet(65536);
    private final Map<Integer, AllocatedPort> allocatedPorts = new HashMap<Integer, AllocatedPort>();
    private final ReferenceQueue<PortRef> dereferencedPorts = new ReferenceQueue();
    private final SystemLevelLocker systemLevelLocker = new SystemLevelLocker();
    private final int assignablePortCount;

    public static PortManager getInstance() {
        return INSTANCE;
    }

    private PortManager() {
        this.portMap.set(0, 1025);
        EphemeralPorts.Range range = EphemeralPorts.getRange();
        this.portMap.set(range.getLower(), range.getUpper() + 1);
        this.reservedPorts.or(this.portMap);
        this.assignablePortCount = 65536 - this.portMap.cardinality();
        if (this.assignablePortCount < 1024) {
            LOGGER.warn("\n*****************************************************************************************\nOnly {} ports available for assignment (ephemeral range=[{}]); operations may be unstable\n*****************************************************************************************", (Object)this.assignablePortCount, (Object)range);
        }
    }

    public synchronized Optional<PortRef> reserve(int port) {
        if (port < 0 || port > 65535) {
            throw new IllegalArgumentException("Port " + port + " is not a valid port number");
        }
        if (this.reservedPorts.get(port)) {
            throw new IllegalArgumentException("Port " + port + " is not reservable");
        }
        this.cleanReleasedPorts();
        if (this.portMap.get(port)) {
            LOGGER.trace("Port {} is already reserved", (Object)port);
            return Optional.empty();
        }
        return Optional.ofNullable(this.reserveInternal(port));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<PortRef> reservePorts(int portCount) {
        if (portCount <= 0 || portCount > this.assignablePortCount) {
            throw new IllegalArgumentException("portCount " + portCount + " not valid");
        }
        ArrayList<PortRef> ports = new ArrayList<PortRef>(portCount);
        try {
            for (int i = 0; i < portCount; ++i) {
                ports.add(this.reservePort());
            }
            List<PortRef> list = Collections.unmodifiableList(ports);
            return list;
        }
        finally {
            if (ports.size() < portCount) {
                ports.forEach(r -> {
                    try {
                        r.close();
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                });
            }
        }
    }

    public synchronized PortRef reservePort() {
        int startingPoint;
        this.cleanReleasedPorts();
        while (this.reservedPorts.get(startingPoint = this.rnd.nextInt(65536))) {
        }
        LOGGER.trace("Starting port reservation search at {}", (Object)startingPoint);
        BitSearch bitSearch = this.rnd.nextBoolean() ? BitSearch.ASCENDING : BitSearch.DESCENDING;
        boolean switched = false;
        int candidatePort = startingPoint;
        while (true) {
            if ((candidatePort = bitSearch.nextFree(this.portMap).applyAsInt(candidatePort)) == -1 && !switched) {
                bitSearch = bitSearch.reverse();
                switched = true;
                candidatePort = startingPoint;
                candidatePort = bitSearch.nextFree(this.portMap).applyAsInt(candidatePort);
            }
            if (candidatePort == -1) {
                throw new IllegalStateException("No port available for reservation");
            }
            PortRef portRef = this.reserveInternal(candidatePort);
            if (portRef != null) {
                return portRef;
            }
            candidatePort = bitSearch.successor(candidatePort);
        }
    }

    private void cleanReleasedPorts() {
        Reference<PortRef> reference;
        while ((reference = this.dereferencedPorts.poll()) != null) {
            if (reference instanceof AllocatedPort) {
                AllocatedPort allocatedPort = (AllocatedPort)reference;
                LOGGER.trace("Port {} dereferenced; releasing reservation", (Object)allocatedPort.port);
                allocatedPort.release();
                continue;
            }
            LOGGER.warn("Unexpected Reference observed: {}", reference);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private PortRef reserveInternal(int candidatePort) {
        LOGGER.trace("Vetting port {}", (Object)candidatePort);
        PortRef portRef = new PortRef(candidatePort);
        boolean releaseRequired = true;
        try {
            boolean systemLevelLockHeld;
            this.portMap.set(candidatePort);
            this.allocatedPorts.put(portRef.port, new AllocatedPort(portRef, this.dereferencedPorts));
            portRef.onClose(() -> this.release(candidatePort));
            try (ServerSocket ignored = new ServerSocket(candidatePort);){
                systemLevelLockHeld = this.systemLevelLocker.lock(portRef);
            }
            if (systemLevelLockHeld) {
                if (this.refusesConnect(candidatePort)) {
                    LOGGER.debug("Reserved port {}", (Object)candidatePort);
                    releaseRequired = false;
                    PortRef portRef2 = portRef;
                    return portRef2;
                }
                LOGGER.trace("Port {} failed refusesConnect", (Object)candidatePort);
            } else {
                LOGGER.trace("Port {} failed to obtain system-level lock", (Object)candidatePort);
            }
        }
        catch (IllegalStateException e) {
            LOGGER.error("Failed to reserve port {}; abandoning reservation", (Object)candidatePort, (Object)e);
            throw e;
        }
        catch (BindException e) {
        }
        catch (Exception e) {
            LOGGER.warn("Failed to reserve port {}; checking next port", (Object)candidatePort, (Object)e);
        }
        finally {
            if (releaseRequired) {
                try {
                    portRef.close();
                }
                catch (Exception e) {}
            }
        }
        return null;
    }

    private synchronized void release(int port) {
        this.portMap.clear(port);
        this.allocatedPorts.remove(port);
        LOGGER.trace("Released JVM-level reservation for port {}", (Object)port);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean refusesConnect(int port) {
        boolean isFree;
        InetSocketAddress endpoint = LOCALHOST == null ? new InetSocketAddress("localhost", port) : new InetSocketAddress(LOCALHOST, port);
        Socket sock = new Socket();
        try {
            sock.connect(endpoint, 50);
            isFree = false;
        }
        catch (ConnectException | SocketTimeoutException e) {
            isFree = true;
        }
        catch (Exception e) {
            LOGGER.debug("[refusesConnect] Client connection to port {} failed for unexpected reason", (Object)endpoint, (Object)e);
            isFree = false;
        }
        finally {
            try {
                sock.close();
            }
            catch (IOException e) {}
        }
        return isFree;
    }

    private static Runnable combine(Runnable a, Runnable b) {
        Objects.requireNonNull(a, "a");
        Objects.requireNonNull(b, "b");
        return () -> {
            try {
                b.run();
            }
            finally {
                a.run();
            }
        };
    }

    static {
        InetAddress localHost = null;
        try {
            localHost = InetAddress.getByName("localhost");
        }
        catch (UnknownHostException e) {
            LOGGER.warn("Unable to obtain an InetAddress for localhost via InetAddress.getByName(\"localhost\")", (Throwable)e);
        }
        LOCALHOST = localHost;
    }

    private static enum BitSearch {
        ASCENDING{

            @Override
            IntUnaryOperator nextFree(BitSet bitSet) {
                return fromIndex -> {
                    int nextClearBit = bitSet.nextClearBit(fromIndex);
                    return nextClearBit > 65535 ? -1 : nextClearBit;
                };
            }

            @Override
            int successor(int value) {
                return value + 1;
            }

            @Override
            BitSearch reverse() {
                return DESCENDING;
            }
        }
        ,
        DESCENDING{

            @Override
            IntUnaryOperator nextFree(BitSet bitSet) {
                return bitSet::previousClearBit;
            }

            @Override
            int successor(int value) {
                return value - 1;
            }

            @Override
            BitSearch reverse() {
                return ASCENDING;
            }
        };


        abstract IntUnaryOperator nextFree(BitSet var1);

        abstract int successor(int var1);

        abstract BitSearch reverse();
    }

    public static class PortRef
    implements AutoCloseable {
        private final int port;
        private final AtomicReference<Runnable> closers = new AtomicReference<Runnable>(() -> {});

        private PortRef(int port) {
            this.port = port;
        }

        void onClose(Runnable action) {
            this.closers.accumulateAndGet(action, (x$0, x$1) -> PortManager.combine(x$0, x$1));
        }

        private AtomicReference<Runnable> closers() {
            return this.closers;
        }

        public int port() {
            return this.port;
        }

        @Override
        public void close() {
            Optional.ofNullable(this.closers.getAndSet(null)).ifPresent(Runnable::run);
        }
    }

    private static class AllocatedPort
    extends WeakReference<PortRef> {
        private final int port;
        private final AtomicReference<Runnable> closer;

        private AllocatedPort(PortRef referent, ReferenceQueue<? super PortRef> q) {
            super(referent, q);
            this.port = referent.port();
            this.closer = referent.closers();
        }

        private void release() {
            try {
                Optional.ofNullable(this.closer.get()).ifPresent(Runnable::run);
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }
}

