/**
 * Copyright (C) 2008 Ovea <dev@testatoo.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.testatoo.config.testatoo;

import com.thoughtworks.selenium.Selenium;
import org.testatoo.config.cartridge.EvaluatorListener;
import org.testatoo.config.cartridge.EvaluatorListenerAdapter;
import org.testatoo.config.cartridge.TestatooCartridge;
import org.testatoo.config.cartridge.TestatooEvaluator;
import org.testatoo.config.selenium.SeleniumSessionConfig;
import org.testatoo.config.selenium.SeleniumSessionConfigBuilder;

import java.util.ArrayList;
import java.util.List;

import static org.testatoo.config.testatoo.Ensure.*;

final class DefaultSeleniumSessionConfigBuilder implements SeleniumSessionConfigBuilder {

    private final List<EvaluatorListener<Selenium>> listeners = new ArrayList<EvaluatorListener<Selenium>>();
    private final DefaultSeleniumSessionConfig seleniumSessionConfig;

    private int timeout = 60000;

    DefaultSeleniumSessionConfigBuilder(DefaultSeleniumSessionConfig seleniumSessionConfig) {
        this.seleniumSessionConfig = seleniumSessionConfig;
    }

    @Override
    public SeleniumSessionConfigBuilder withTimeout(int timeoutMilliseconds) {
        this.timeout = timeoutMilliseconds;
        return this;
    }

    @Override
    public SeleniumSessionConfigBuilder add(EvaluatorListener<Selenium> listener) {
        notNull(listener, "Selenium session listener");
        listeners.add(listener);
        return this;
    }

    @SuppressWarnings({"unchecked"})
    @Override
    public SeleniumSessionConfig inCartridge(final TestatooCartridge cartridge) {
        listeners.add(new EvaluatorListenerAdapter<Selenium>() {
            @Override
            public void beforeStart(Selenium session) {
                CartridgeFactory.<Selenium>get(cartridge, TestatooEvaluator.from(session)).configure(session);
            }

            @Override
            public void afterStart(Selenium session) {
                session.setTimeout("" + timeout);
            }
        });
        return seleniumSessionConfig;
    }

    void fireAfterStart(Selenium session) {
        for (EvaluatorListener<Selenium> listener : listeners)
            listener.afterStart(session);
    }

    void fireBeforeStart(Selenium session) {
        for (EvaluatorListener<Selenium> listener : listeners)
            listener.beforeStart(session);
    }

    void fireBeforeStop(Selenium session) {
        for (EvaluatorListener<Selenium> listener : listeners)
            listener.beforeStop(session);
    }

    void fireAfterStop(Selenium session) {
        for (EvaluatorListener<Selenium> listener : listeners)
            listener.afterStop(session);
    }
}
