/**
 * Copyright (C) 2008 Ovea <dev@testatoo.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.testatoo.config.testatoo;

import com.mycila.log.Logger;
import com.mycila.log.Loggers;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.testatoo.config.TestatooConfig;
import org.testatoo.config.TestatooModule;
import org.testatoo.config.annotation.Implementation;
import org.testatoo.config.container.ContainerBuilder;
import org.testatoo.config.container.ContainerConfig;
import org.testatoo.config.container.ContainerType;
import org.testatoo.config.lifecycle.LifeCycleConfig;
import org.testatoo.config.lifecycle.TestListener;
import org.testatoo.config.selenium.SeleniumServerBuilder;
import org.testatoo.config.selenium.SeleniumServerConfig;
import org.testatoo.config.selenium.SeleniumSessionBuilder;
import org.testatoo.config.selenium.SeleniumSessionConfig;
import org.testatoo.container.TestatooContainer;
import org.testatoo.core.Current;
import org.testatoo.core.EvaluatorHolder;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import static org.testatoo.config.testatoo.Ensure.*;

final class DefaultTestatooConfig implements TestatooConfig {

    private static final Logger LOGGER = Loggers.get(DefaultTestatooConfig.class);

    private final List<EventListener> listeners = new ArrayList<EventListener>(100);
    private final DefaultContainerConfig containerConfig;
    private final DefaultSeleniumServerConfig seleniumServerConfig;
    private final DefaultLifeCycleConfig lifeCycleConfig;
    private final DefaultSeleniumSessionConfig seleniumSessionConfig;
    private MethodInterceptor interceptor = new MethodInterceptor() {
        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            return invocation.proceed();
        }
    };
    private final List<TestListener> testListeners = new LinkedList<TestListener>();

    DefaultTestatooConfig() {
        containerConfig = new DefaultContainerConfig(this);
        seleniumServerConfig = new DefaultSeleniumServerConfig(this);
        lifeCycleConfig = new DefaultLifeCycleConfig(this);
        seleniumSessionConfig = new DefaultSeleniumSessionConfig(this);
    }

    @Override
    public TestatooConfig install(TestatooModule... modules) {
        notNull(modules, "Testatoo module list");
        for (TestatooModule module : modules) {
            module.configure(this);
        }
        return this;
    }

    @Override
    public TestatooConfig useAnnotations() {
        add(new TestListener() {
            @Override
            public void onTest(Object instance, Method method) {
                Class<?> c = instance.getClass();
                while (c != null && !c.equals(Object.class)) {
                    for (Field field : c.getDeclaredFields()) {
                        Implementation implementation = field.getAnnotation(Implementation.class);
                        if (implementation != null) {
                            field.setAccessible(true);
                            try {
                                field.set(instance, new CurrentImpl());
                            } catch (IllegalAccessException e) {
                                throw new RuntimeException(e.getMessage(), e);
                            }
                        }
                    }
                    c = c.getSuperclass();
                }
            }
        });
        return this;
    }

    @Override
    public LifeCycleConfig lifecycle() {
        return lifeCycleConfig;
    }

    @Override
    public ContainerConfig containers() {
        return containerConfig;
    }

    @Override
    public ContainerType createContainer() {
        return new ContainerType() {
            @Override
            public ContainerBuilder implementedBy(TestatooContainer type) {
                notNull(type, "Container type");
                return implementedBy(type.toString());
            }

            @Override
            public ContainerBuilder implementedBy(String containerClass) {
                notNull(containerClass, "Container class");
                return new DefaultContainerBuilder(containerClass);
            }
        };
    }

    @Override
    public SeleniumServerConfig seleniumServers() {
        return seleniumServerConfig;
    }

    @Override
    public SeleniumServerBuilder createSeleniumServer() {
        return new DefaultSeleniumServerBuilder();
    }

    @Override
    public SeleniumSessionConfig seleniumSessions() {
        return seleniumSessionConfig;
    }

    @Override
    public SeleniumSessionBuilder createSeleniumSession() {
        return new DefaultSeleniumSessionBuilder();
    }

    void register(EventListener eventListener) {
        listeners.add(eventListener);
    }

    void fire(Event event) {
        LOGGER.debug("Firing %s event...", event);
        for (EventListener listener : event.ordering().sort(new ArrayList<EventListener>(listeners))) {
            listener.on(event);
        }
    }

    public void add(TestListener testListener) {
        testListeners.add(testListener);
    }

    public void fireExecution(Object o, Method m) {
        for (TestListener testListener : testListeners) {
            testListener.onTest(o, m);
        }
    }

    public void register(MethodInterceptor interceptor) {
        this.interceptor = interceptor;
    }

    public void fire(MethodInvocation testInvocation) throws Throwable {
        interceptor.invoke(testInvocation);
    }

    private static final class CurrentImpl implements Current {
        @Override
        public Object get() {
            return EvaluatorHolder.get().implementation();
        }
    }
}
