package org.tharos.jdbc.swissknife.core;

import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.tharos.jdbc.swissknife.dto.Column;
import org.tharos.jdbc.swissknife.dto.Table;

/**
 * Utility class for jdbc metadata extraction
 */
public class DatabaseMetadataExtractor {

  private Logger LOGGER = LogManager.getLogger(getClass().getName());

  private String className, URL, user, password;
  private Connection connection;

  private String schema;

  /**
   * Default constructor
   * @param className The jdbc driver's class name
   * @param URL The database URL in jdbc compliant format
   * @param user The database user
   * @param password The database password
   * @param schema The database schema
   */
  public DatabaseMetadataExtractor(
    String className,
    String URL,
    String user,
    String password,
    String schema
  ) {
    this.className = className;
    this.URL = URL;
    this.user = user;
    this.password = password;
    this.connection = null;
    this.schema = schema;
  }

  /**
   * Method for initializing a database connection
   * @return An established connection to the database
   */
  private Connection initializeConnection() {
    try {
      Class.forName(className);
    } catch (ClassNotFoundException ex) {
      LOGGER.error("Unable to load the class. Terminating the program");
      System.exit(-1);
    }
    try {
      connection = DriverManager.getConnection(URL, user, password);
    } catch (SQLException ex) {
      LOGGER.error("Error getting connection: " + ex.getMessage());
      System.exit(-1);
    } catch (Exception ex) {
      LOGGER.error("Error: " + ex.getMessage());
      System.exit(-1);
    }
    if (connection != null) {
      LOGGER.info("Connected Successfully!");
    }
    return connection;
  }

  /**
   * Method for the extraction of the accessible database table list
   * @return A List of Table(s)
   */
  public List<Table> getTablesList() {
    if (connection == null) {
      initializeConnection();
    }
    DatabaseMetaData databaseMetaData = null;
    ArrayList<Table> tableList = new ArrayList<Table>();
    try {
      databaseMetaData = connection.getMetaData();
      String[] tableType = new String[] { "TABLE" };
      ResultSet resultSet = databaseMetaData.getTables(
        null,
        this.schema,
        null,
        tableType
      );
      while (resultSet.next()) {
        Table tbl = new Table();
        tbl.setSchemaName(this.schema);
        tbl.setName(resultSet.getString("TABLE_NAME"));
        tbl.setSequenceName("seq_" + tbl.getName());
        tbl.setColumnList(this.extractColumnsInfo(tbl.getName()));
        tableList.add(tbl);
      }
    } catch (SQLException ex) {
      LOGGER.error(
        "Error while fetching metadata. Terminating program.. " +
        ex.getMessage()
      );
      System.exit(-1);
    } catch (Exception ex) {
      LOGGER.error(
        "Error while fetching metadata. Terminating program.. " +
        ex.getMessage()
      );
      System.exit(-1);
    } finally {
      closeConnection();
    }

    return tableList;
  }

  /**
   * Utility method for safely closing the connection to the database
   */
  private void closeConnection() {
    try {
      this.connection.close();
    } catch (SQLException e) {
      try {
        if (!this.connection.isClosed()) {
          this.connection.close();
          this.connection = null;
        }
      } catch (Exception ex) {
        LOGGER.error("Cannot close connection. Exit.");
        this.connection = null;
        System.exit(-1);
      }
    }
  }

  /**
   * Utility method for the extraction of columns' accessible metadata for the input table
   * @param tableName The name of the table to be analyzed
   * @return A List of Column(s)
   */
  public List<Column> extractColumnsInfo(String tableName) {
    if (connection == null) {
      initializeConnection();
    }
    Map<String, Column> colsMap = new HashMap<String, Column>();
    try {
      DatabaseMetaData databaseMetaData = connection.getMetaData();
      ResultSet columns = databaseMetaData.getColumns(
        null,
        schema,
        tableName,
        null
      );
      while (columns.next()) {
        Column col = new Column();
        col.setName(columns.getString("COLUMN_NAME"));
        col.setSqlType(columns.getInt("DATA_TYPE"));
        col.setType(SQLTypeMap.toClass(columns.getInt("DATA_TYPE")));
        col.setNullable(columns.getBoolean("IS_NULLABLE"));
        col.setAutoincrement(columns.getBoolean("IS_AUTOINCREMENT"));
        col.setSize(columns.getInt("COLUMN_SIZE"));
        col.setDecimalDigits(columns.getInt("DECIMAL_DIGITS"));
        colsMap.put(col.getName(), col);
      }
      ResultSet PK = databaseMetaData.getPrimaryKeys(null, schema, tableName);
      while (PK.next()) {
        String pkCol = PK.getString("COLUMN_NAME");
        if (colsMap.containsKey(pkCol)) {
          colsMap.get(pkCol).setPrimaryKey(true);
        }
      }

      // Get Foreign Keys
      ResultSet FK = databaseMetaData.getImportedKeys(null, schema, tableName);
      while (FK.next()) {
        String fkTable = FK.getString("FKTABLE_NAME");
        String fkColumn = FK.getString("FKCOLUMN_NAME");
        String ownColumn = FK.getString("PKCOLUMN_NAME");
        if (colsMap.containsKey(ownColumn)) {
          colsMap.get(ownColumn).setForeignTableName(fkTable);
          colsMap.get(ownColumn).setForeignColumnName(fkColumn);
        }
      }
    } catch (SQLException ex) {
      LOGGER.error(
        "Error while fetching metadata. Terminating program.. " +
        ex.getMessage()
      );
      System.exit(-1);
    } catch (Exception ex) {
      LOGGER.error(
        "Error while fetching metadata. Terminating program.. " +
        ex.getMessage()
      );
      System.exit(-1);
    }
    return new ArrayList<Column>(colsMap.values());
  }
}
