package org.thewonderlemming.c4plantuml.graphml.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;

import org.thewonderlemming.c4plantuml.graphml.model.builder.KeyFor;

/**
 * Holds C4 GraphML keys and their properties.
 *
 * @author thewonderlemming
 *
 */
public enum C4Keys {

    /**
     * The graph's C4 aspect.
     */
    ASPECT("attr00", KeyFor.GRAPH, "aspect", C4Keys.STRING_TYPE),

    /**
     * The C4 level the graph's edge belongs to.
     */
    C4_LEVEL("attr01", KeyFor.EDGE, "c4_level", C4Keys.STRING_TYPE),

    /**
     * The graph's node description.
     */
    DESCRIPTION("attr02", KeyFor.NODE, "description", C4Keys.STRING_TYPE),

    /**
     * The graph's node type (i.e. Person, System, and so on).
     */
    ENTITY_TYPE("attr03", KeyFor.NODE, "entity_type", C4Keys.STRING_TYPE),

    /**
     * The label on a graph's edge.
     */
    LABEL("attr05", KeyFor.EDGE, "label", C4Keys.STRING_TYPE),

    /**
     * The graph's node name.
     */
    NAME("attr06", KeyFor.NODE, "name", C4Keys.STRING_TYPE),

    /**
     * The protocol on a graph's edge (HTTPS, OAuth2, and such)..
     */
    PROTOCOL("attr07", KeyFor.EDGE, "protocol", C4Keys.STRING_TYPE),

    /**
     * The technological stack of a graph's node.
     */
    TECHNOLOGICAL_STACK("attr08", KeyFor.NODE, "technological_stack", C4Keys.STRING_TYPE),

    /**
     * The graph's title.
     */
    TITLE("attr09", KeyFor.GRAPH, "title", C4Keys.STRING_TYPE);


    private static final List<KeyModel> KEYS = new ArrayList<>();

    private static final String STRING_TYPE = "string";

    private final String attrName;

    private final String attrType;

    private final String id;

    private final KeyFor keyFor;


    static {
        Stream
            .of(C4Keys.values())
                .forEach(value -> KEYS.add(value.getC4Key()));
    }


    /**
     * Returns a {@link List} of {@link KeyModel} built from every value from {@link C4Keys}.
     *
     * @return a list of {@link KeyModel} built from every instance of the current class.
     */
    public static List<KeyModel> getC4Keys() {
        return Collections.unmodifiableList(KEYS);
    }

    private C4Keys(final String id, final KeyFor keyFor, final String attrName, final String attrType) {

        this.id = id;
        this.keyFor = keyFor;
        this.attrName = attrName;
        this.attrType = attrType;
    }

    /**
     * Builds a {@link KeyModel} from the current {@link C4Keys} and returns it..
     *
     * @return a {@link KeyModel} instance built from the current {@link C4Keys} value.
     */
    public KeyModel getC4Key() {

        return KeyModel
            .builder()
                .withId(this.id)
                .withFor(this.keyFor)
                .withAttrName(this.attrName)
                .withAttrType(this.attrType)
                .build();
    }

    /**
     * Return the GraphML key ID for the current {@link C4Keys} value.
     *
     * @return the current GraphML key ID.
     */
    public String getId() {
        return this.id;
    }
}
