package org.thewonderlemming.c4plantuml.graphml.model;

import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.thewonderlemming.c4plantuml.graphml.Build;

/**
 * The root node of a Java representation of a GraphML stream.
 * <p>
 * It is based on JAXB.
 *
 * @author thewonderlemming
 *
 */
@XmlRootElement(name = "graphml")
@XmlAccessorType(XmlAccessType.FIELD)
public class GraphMLModel {

    /**
     * The GraphML namespace.
     */
    public static final String NAMESPACE = "http://graphml.graphdrawing.org/xmlns";

    /**
     * The GraphML schema location.
     */
    public static final String SCHEMA_LOCATION = "http://graphml.graphdrawing.org/xmlns"
        + " http://graphml.graphdrawing.org/xmlns/1.1/graphml.xsd";

    @XmlElement(name = KeyModel.TAG_NAME, required = true)
    private List<KeyModel> seq1Keys;

    @XmlElement(name = GraphModel.TAG_NAME, required = true)
    private MappedListDecorator<GraphModel, String> seq2Graphs;


    /**
     * A builder to the current {@link GraphMLModel} class.
     *
     * @return a new {@link GraphMLModel} instance.
     */
    public static WithKeys<WithGraphs<Build<GraphMLModel>>> builder() {

        return keys -> graphs -> () -> {

            final GraphMLModel graphMl = new GraphMLModel();
            graphMl.setKeys(keys);
            graphMl.setGraphs(graphs);

            return graphMl;
        };
    }

    /**
     * Default constructor.
     */
    private GraphMLModel() {
        // Does nothing but hiding the current constructor.
    }

    /**
     * Adds a {@link GraphModel} to the current model.
     *
     * @param graph the {@link GraphModel} to add to the current instance.
     */
    public void addGraph(final GraphModel graph) {
        this.seq2Graphs.addOrReplaceData(graph);
    }

    /**
     * Sets a list of {@link GraphModel} in the current instance.
     *
     * @param graphs a list of graphs to add to the current instance.
     */
    private void setGraphs(final List<GraphModel> graphs) {
        this.seq2Graphs = new MappedListDecorator<>(
            GraphModel.class,
                graphs,
                GraphModel::getId);
    }

    /**
     * Sets a list of {@link KeyModel} in the current instance.
     *
     * @param keys a list of keys to add to the current instance.
     */
    private void setKeys(final List<KeyModel> keys) {
        this.seq1Keys = keys;
    }
}
