package org.thewonderlemming.c4plantuml.graphml.validation;

import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

/**
 * A DOM {@link LSResourceResolver} implementation to avoid having external calls to retrieve GraphML related XSD files.
 *
 * @author thewonderlemming
 *
 */
public class CustomResourceResolver implements LSResourceResolver {

    private static final Logger LOGGER = LoggerFactory.getLogger(CustomResourceResolver.class);


    /**
     * {@inheritDoc}
     * <p>
     * Returns a {@link CustomLSInput} with the managed {@link GraphMLXsd} if found, or null, else.
     */
    @Override
    public LSInput resolveResource(final String type, final String namespaceURI, final String publicId,
        final String systemId, final String baseURI) {

        final Optional<GraphMLXsd> resolvedXsd = GraphMLXsd.resolveXsd(systemId, namespaceURI);

        if (!resolvedXsd.isPresent()) {

            LOGGER
                .warn(
                    "Could not resolve locally the resource matching the following arguments: "
                        + "type='{}', "
                        + "namespaceURI='{}', "
                        + "publicId='{}', "
                        + "systemId='{}', "
                        + "baseURI='{}'",
                    type,
                    namespaceURI,
                    publicId,
                    systemId,
                    baseURI);

            return null;
        }

        return CustomLSInput
            .newBuilder()
                .withBaseURI(baseURI)
                .withPublicId(publicId)
                .withResourceFilename(resolvedXsd.get().getResourceFilename())
                .withSystemId(systemId)
                .build();
    }
}
