package org.thewonderlemming.c4plantuml.graphml.validation;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * A custom implementation of {@link ErrorHandler} to log validation errors and warnings, then to throw exceptions if
 * wanted.
 *
 * @author thewonderlemming
 *
 */
public class GraphMLValidatorErrorHandler implements ErrorHandler {

    private final boolean strictValidation;


    /**
     * Default constructor.
     *
     * @param strictValidation throws exceptions on validation errors and warnings if set to {@code true}.
     */
    public GraphMLValidatorErrorHandler(final boolean strictValidation) {
        this.strictValidation = strictValidation;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void error(final SAXParseException exception) throws SAXException {
        report(exception, (logger, msg, e) -> logger.error(msg, e));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void fatalError(final SAXParseException exception) throws SAXException {
        report(exception, (logger, msg, e) -> logger.error(msg, e));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void warning(final SAXParseException exception) throws SAXException {
        report(exception, (logger, msg, e) -> logger.warn(msg, e));
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(GraphMLValidator.class);
    }

    private void report(final SAXParseException exception, final TripleConsumer<Logger, String, Throwable> logger)
        throws SAXException {

        logger.accept(getLogger(), exception.getMessage(), exception);

        if (strictValidation) {
            throw new ValidationException(exception.getMessage(), exception);
        }
    }
}
