package org.thewonderlemming.c4plantuml.linter;

import java.util.HashSet;
import java.util.Set;

import org.antlr.v4.runtime.BaseErrorListener;
import org.antlr.v4.runtime.RecognitionException;
import org.antlr.v4.runtime.Recognizer;
import org.thewonderlemming.c4plantuml.linter.rules.AbstractLintingRule;

/**
 * An observer pattern to enable {@link AbstractLintingRule} to be notified on syntax errors.
 *
 * @author thewonderlemming
 *
 */
public class LinterSyntaxErrorListener extends BaseErrorListener {

    private final Set<AbstractLintingRule> rules = new HashSet<>();


    /**
     * registers an {@link AbstractLintingRule} so that it can be notified of syntax errors.
     *
     * @param rule the rule to register.
     */
    public void registerRule(final AbstractLintingRule rule) {
        this.rules.add(rule);
    }

    /**
     * Reports a parsing syntax error to the registered rules.
     */
    @Override
    public void syntaxError(final Recognizer<?, ?> recognizer, final Object offendingSymbol, final int line,
        final int charPositionInLine, final String msg, final RecognitionException e) {

        final SyntaxError error = new SyntaxError(recognizer, offendingSymbol, line, charPositionInLine, msg, e);
        this.rules.forEach(rule -> rule.reportSyntaxError(error));
    }

    /**
     * Unregister a {@link AbstractLintingRule} so that it won't be notified of syntax errors anymore.
     *
     * @param rule the rule to unregister.
     */
    public void unregisterRule(final AbstractLintingRule rule) {
        this.rules.remove(rule);
    }
}
