package org.thewonderlemming.c4plantuml.linter;

import org.antlr.v4.runtime.RecognitionException;
import org.antlr.v4.runtime.Recognizer;
import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Represents a parsing syntax error and holds information such as the line of the error, the offending symbol and so
 * on.
 *
 * @author thewonderlemming
 *
 */
public class SyntaxError {

    private final int charPositionInLine;

    private final RecognitionException e;

    private final int line;

    private final String msg;

    private final Object offendingSymbol;

    private final Recognizer<?, ?> recognizer;


    /**
     * Default constructor.
     *
     * @param recognizer the ANTLR4 object that recognizes the input and handles the errors.
     * @param offendingSymbol the unexpected symbol that caused the syntax error.
     * @param line the line of the syntax error in the source file.
     * @param charPositionInLine the position of the offending symbol in the error line.
     * @param msg the error message.
     * @param e the error {@link RecognitionException}.
     */
    public SyntaxError(final Recognizer<?, ?> recognizer, final Object offendingSymbol, final int line,
        final int charPositionInLine,
        final String msg, final RecognitionException e) {

        this.recognizer = recognizer;
        this.offendingSymbol = offendingSymbol;
        this.line = line;
        this.charPositionInLine = charPositionInLine;
        this.msg = msg;
        this.e = e;
    }

    /**
     * Returns the position of the offending character in the error line.
     *
     * @return the position of the offending character.
     */
    public int getCharPositionInLine() {
        return charPositionInLine;
    }

    /**
     * returns the error associated exception.
     *
     * @return the current exception.
     */
    public RecognitionException getE() {
        return e;
    }

    /**
     * Returns the line of the error in the source file.
     *
     * @return the line of the error.
     */
    public int getLine() {
        return line;
    }

    /**
     * returns the message of the syntax error.
     *
     * @return the error message.
     */
    public String getMsg() {
        return msg;
    }

    /**
     * Returns the unexpected symbol that caused that error.
     *
     * @return the unexpected symbol.
     */
    public Object getOffendingSymbol() {
        return offendingSymbol;
    }

    /**
     * Returns the ANTLR4 object that recognizes the input and handles the errors.
     *
     * @return the ANTLR4 recognizer instance.
     */
    public Recognizer<?, ?> getRecognizer() {
        return recognizer;
    }

    /**
     * Returns a human understandable representation of the current instance.
     */
    @Override
    public String toString() {

        return new ToStringBuilder(this)
            .append("charPositionInLine", this.charPositionInLine)
                .append("line", this.line)
                .append("offendingSymbol", this.offendingSymbol)
                .append("message", this.msg)
                .append("exception", this.e)
                .toString();
    }
}
