package org.thewonderlemming.c4plantuml.linter.rules.builtin;

import java.util.Optional;

import org.antlr.v4.runtime.Parser;
import org.antlr.v4.runtime.tree.ParseTreeListener;
import org.thewonderlemming.c4plantuml.commons.Reporter;
import org.thewonderlemming.c4plantuml.linter.rules.RuleParameters;

/**
 * A linting rule that verifies that every alias is referenced in a given dictionary file.
 *
 * @author thewonderlemming
 *
 */
public class AliasesShouldBeListedInDictionaryRule extends AbstractC4Rule {

    /**
     * The default dictionary filename if no filename is passed as a parameter to the rule..
     */
    public static final String ALIAS_DEFAULT_DICT_FILENAME = "";

    private final String aliasDictionaryFilename;


    /**
     * Returns the name of the parameter that needs to be set to override the default value of the dictionary filename
     * {@link AliasesShouldBeListedInDictionaryRule#ALIAS_DEFAULT_DICT_FILENAME}.
     *
     * @return the name of the parameter that controls the filename value.
     */
    public static String getAliasDictionaryFilenameParameterName() {
        return AliasesShouldBeListedInDictionaryRule.class.getSimpleName() + ".aliasDictionaryFilename";
    }

    /**
     * Default constructor.
     *
     * @param parameters the {@link RuleParameters} that are passed to control the behavior of this rule.
     */
    public AliasesShouldBeListedInDictionaryRule(final RuleParameters parameters) {

        super(parameters);

        final Optional<String> optionalAliasFormat = parameters.getParameter(getAliasDictionaryFilenameParameterName());
        this.aliasDictionaryFilename = optionalAliasFormat.isPresent()
            ? optionalAliasFormat.get()
            : ALIAS_DEFAULT_DICT_FILENAME;
    }

    /**
     * {@inheritDoc}
     * <p>
     * Creates a {@link AliasesShouldBeListedInDictionaryListener} instance as a listener.
     */
    @Override
    public Optional<? extends ParseTreeListener> createParseTreeListener(final Reporter reporter,
        final Class<? extends Parser> parserType) {

        return this.acceptableParsersTypes().contains(parserType)
            ? Optional.of(new AliasesShouldBeListedInDictionaryListener(this, reporter, this.aliasDictionaryFilename))
            : Optional.empty();
    }
}
