package org.thewonderlemming.c4plantuml.linter.rules.builtin;

import java.util.Optional;

import org.antlr.v4.runtime.Parser;
import org.antlr.v4.runtime.tree.ParseTreeListener;
import org.thewonderlemming.c4plantuml.commons.Reporter;
import org.thewonderlemming.c4plantuml.linter.rules.RuleParameters;

/**
 * A linting rule that checks that the aliases comply to a given regular expression.
 *
 * @author thewonderlemming
 *
 */
public class AliasesShouldFollowStandardConventionRule extends AbstractC4Rule {

    /**
     * The default regular expression to use for the aliases if no format is passed as a parameter of the rule.
     */
    public static final String ALIAS_DEFAULT_FORMAT = "^[a-zA-Z_-]+$";

    private final String aliasFormat;


    /**
     * Returns the name of the parameter that needs to be set to override the default value of the aliases regular
     * expression {@link AliasesShouldFollowStandardConventionRule#ALIAS_DEFAULT_FORMAT}.
     *
     * @return the name of the parameter that controls the aliases regular expression value.
     */
    public static String getAliasFormatParameterName() {
        return AliasesShouldFollowStandardConventionRule.class.getSimpleName() + ".aliasDefaultFormatRegex";
    }

    /**
     * Default constructor.
     *
     * @param parameters the {@link RuleParameters} that are passed to control the behavior of this rule.
     */
    public AliasesShouldFollowStandardConventionRule(final RuleParameters parameters) {

        super(parameters);

        final Optional<String> optionalAliasFormat = parameters.getParameter(getAliasFormatParameterName());
        this.aliasFormat = optionalAliasFormat.isPresent()
            ? optionalAliasFormat.get()
            : ALIAS_DEFAULT_FORMAT;
    }

    /**
     * {@inheritDoc}
     * <p>
     * Creates a {@link AliasesShouldFollowStandardConventionListener} instance as a listener.
     */
    @Override
    public Optional<? extends ParseTreeListener> createParseTreeListener(final Reporter reporter,
        final Class<? extends Parser> parserType) {

        return this.acceptableParsersTypes().contains(parserType)
            ? Optional.of(new AliasesShouldFollowStandardConventionListener(this, reporter, this.aliasFormat))
            : Optional.empty();
    }
}
