package org.threadly.db;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Date;
import java.sql.NClob;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Savepoint;
import java.sql.Statement;
import java.sql.Struct;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Abstract implementation which tries to delegate actions to a contained connection.  This 
 * additionally tries to select the delegated connection at the last possible moment.  As much as 
 * possible the logic for when a delegated connection needs to be rotated out is also done here.
 * <p>
 * This class is not very useful outside of Threadly developers.
 */
public abstract class AbstractDelegatingConnection implements Connection {
  protected abstract <R> R processOnDelegate(SQLOperation<Connection, R> action) throws SQLException;
  
  protected abstract Connection getReferenceConnection();
  
  protected abstract void resetStickyConnection();

  @Override
  public <T> T unwrap(Class<T> iface) throws SQLException {
    // We are sort of a wrapper, but...it's complicated, so we pretend we are not
    try {
      return iface.cast(this);
    } catch (ClassCastException e) {
      throw new SQLException("Unable to unwrap to " + iface, e);
    }
  }

  @Override
  public boolean isWrapperFor(Class<?> iface) throws SQLException {
    return iface.isInstance(this);
  }

  @Override
  public void commit() throws SQLException {
    processOnDelegate((c) -> { c.commit(); return null; });
    // If understood correctly the sticky connection can now be reset to allow connection
    //       cycling even if autoCommit is disabled
    resetStickyConnection();
  }

  @Override
  public void rollback() throws SQLException {
    processOnDelegate((c) -> { c.rollback(); return null; });
    // If understood correctly the sticky connection can now be reset to allow connection
    //       cycling even if autoCommit is disabled
    resetStickyConnection();
  }

  @Override
  public void rollback(Savepoint savepoint) throws SQLException {
    processOnDelegate((c) -> { c.rollback(savepoint); return null; });
    // If understood correctly the sticky connection can now be reset to allow connection
    //       cycling even if autoCommit is disabled
    resetStickyConnection();
  }

  @Override
  public Savepoint setSavepoint() throws SQLException {
    return processOnDelegate((c) -> c.setSavepoint());
  }

  @Override
  public Savepoint setSavepoint(String name) throws SQLException {
    return processOnDelegate((c) -> c.setSavepoint(name));
  }

  @Override
  public void releaseSavepoint(Savepoint savepoint) throws SQLException {
    processOnDelegate((c) -> { c.releaseSavepoint(savepoint); return null; });
  }

  @Override
  public Statement createStatement() throws SQLException {
    return new DelegatingStatement();
  }

  @Override
  public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
    return new DelegatingStatement(resultSetType, resultSetConcurrency);
  }

  @Override
  public Statement createStatement(int resultSetType, int resultSetConcurrency,
                                   int resultSetHoldability) throws SQLException {
    return new DelegatingStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
  }

  @Override
  public PreparedStatement prepareStatement(String sql) throws SQLException {
    return new DelegatingPreparedStatement(sql);
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
    return new DelegatingPreparedStatement(sql, columnIndexes);
  }

  @Override
  public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
    return new DelegatingPreparedStatement(sql, columnNames);
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
    return new DelegatingPreparedStatement(sql, autoGeneratedKeys);
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int resultSetType,
                                            int resultSetConcurrency) throws SQLException {
    return new DelegatingPreparedStatement(sql, resultSetType, resultSetConcurrency);
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency,
                                            int resultSetHoldability) throws SQLException {
    return new DelegatingPreparedStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
  }

  @Override
  public CallableStatement prepareCall(String sql) throws SQLException {
    return new DelegatingCallableStatement(sql);
  }

  @Override
  public CallableStatement prepareCall(String sql, int resultSetType,
                                       int resultSetConcurrency) throws SQLException {
    return new DelegatingCallableStatement(sql, resultSetType, resultSetConcurrency);
  }

  @Override
  public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency,
                                       int resultSetHoldability) throws SQLException {
    return new DelegatingCallableStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
  }

  @Override
  public Clob createClob() throws SQLException {
    return getReferenceConnection().createClob();
  }

  @Override
  public Blob createBlob() throws SQLException {
    return getReferenceConnection().createBlob();
  }

  @Override
  public NClob createNClob() throws SQLException {
    return getReferenceConnection().createNClob();
  }

  @Override
  public SQLXML createSQLXML() throws SQLException {
    return getReferenceConnection().createSQLXML();
  }

  @Override
  public String getClientInfo(String name) throws SQLException {
    return getReferenceConnection().getClientInfo(name);
  }

  @Override
  public Properties getClientInfo() throws SQLException {
    return getReferenceConnection().getClientInfo();
  }

  @Override
  public String getCatalog() throws SQLException {
    return getReferenceConnection().getCatalog();
  }

  @Override
  public int getHoldability() throws SQLException {
    return getReferenceConnection().getHoldability();
  }

  @Override
  public DatabaseMetaData getMetaData() throws SQLException {
    return getReferenceConnection().getMetaData();
  }

  @Override
  public int getNetworkTimeout() throws SQLException {
    return getReferenceConnection().getNetworkTimeout();
  }

  @Override
  public String getSchema() throws SQLException {
    return getReferenceConnection().getSchema();
  }
  
  @Override
  public Map<String, Class<?>> getTypeMap() throws SQLException {
    return getReferenceConnection().getTypeMap();
  }

  @Override
  public Array createArrayOf(String typeName, Object[] elements) throws SQLException {
    return getReferenceConnection().createArrayOf(typeName, elements);
  }

  @Override
  public Struct createStruct(String typeName, Object[] attributes) throws SQLException {
    return getReferenceConnection().createStruct(typeName, attributes);
  }

  @Override
  public String nativeSQL(String sql) throws SQLException {
    return getReferenceConnection().nativeSQL(sql);
  }
  
  /**
   * Small interface for operations which are delegated to a given type and may throw a 
   * {@link SQLException}.  This exception is the only reason for this interface.
   * 
   * @param <T> Type of object operation is ran on
   * @param <R> Type of result returned from operation
   */
  protected interface SQLOperation<T, R> {
    public R run(T obj) throws SQLException;
  }
  
  /**
   * Abstract implementation for how statement's actions are queued and dispatched once created.  
   * Unfortunately this means we need to duplicate code when there is more than one statement deep, 
   * but we can't otherwise extend multiple layers without running into generics conflicts.
   *
   * @param <ST> Type of statement implementation
   */
  protected abstract class AbstractDelegatingStatement<ST extends Statement> implements Statement {
    protected final int resultSetType;
    protected final int resultSetConcurrency;
    protected final int resultSetHoldability;
    private ST delegateStatement = null;
    private List<SQLOperation<ST, Void>> queuedStatementActions = null;
    
    public AbstractDelegatingStatement() throws SQLException {
      this(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
    }

    public AbstractDelegatingStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
      this(resultSetType, resultSetConcurrency, ResultSet.HOLD_CURSORS_OVER_COMMIT);
    }

    protected AbstractDelegatingStatement(int resultSetType, int resultSetConcurrency,
                                          int resultSetHoldability) throws SQLException {
      this.resultSetType = resultSetType;
      this.resultSetConcurrency = resultSetConcurrency;
      this.resultSetHoldability = resultSetHoldability;
      delegateStatement = null;
      queuedStatementActions = null;
    }

    protected abstract SQLOperation<Connection, ST> delegateStatementProvider();
    
    // used whenever there is a result needed
    protected ST delegate() throws SQLException {
      synchronized (this) { // TODO - improve performance
        if (delegateStatement == null) {
          delegateStatement = processOnDelegate(delegateStatementProvider());
          if (queuedStatementActions != null) {
            for (SQLOperation<ST, Void> o : queuedStatementActions) {
              o.run(delegateStatement);
            }
            queuedStatementActions = null;
          }
        }
      }
      
      return delegateStatement;
    }

    // used when there is no result, and thus the action may be delayed
    protected void action(SQLOperation<ST, Void> action) throws SQLException {
      synchronized (this) { // TODO - improve performance
        if (delegateStatement == null) {
          if (queuedStatementActions == null) {
            queuedStatementActions = new ArrayList<>(2);
          }
          queuedStatementActions.add(action);
        } else {
          action.run(delegateStatement);
        }
      }
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
      // We are sort of a wrapper, but...it's complicated, so we pretend we are not
      try {
        return iface.cast(this);
      } catch (ClassCastException e) {
        throw new SQLException("Unable to unwrap to " + iface, e);
      }
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
      return iface.isInstance(this);
    }

    @Override
    public int getResultSetConcurrency() throws SQLException {
      return resultSetConcurrency;
    }

    @Override
    public int getResultSetHoldability() throws SQLException {
      return resultSetHoldability;
    }

    @Override
    public int getResultSetType() throws SQLException {
      return resultSetType;
    }

    @Override
    public void addBatch(String sql) throws SQLException {
      action((s) -> { s.addBatch(sql); return null; });
    }

    @Override
    public void cancel() throws SQLException {
      if (delegateStatement != null) {
        // only run if statement has been produced
        delegateStatement.cancel();
      }
    }

    @Override
    public void clearBatch() throws SQLException {
      action((s) -> { s.clearBatch(); return null; });
    }

    @Override
    public void clearWarnings() throws SQLException {
      action((s) -> { s.clearWarnings(); return null; });
    }

    @Override
    public void close() throws SQLException {
      // close needs to be processed immediately or it may never run
      delegate().close();
    }

    @Override
    public void closeOnCompletion() throws SQLException {
      action((s) -> { s.closeOnCompletion(); return null; });
    }

    @Override
    public boolean execute(String sql) throws SQLException {
      return delegate().execute(sql);
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
      return delegate().execute(sql, autoGeneratedKeys);
    }

    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
      return delegate().execute(sql, columnIndexes);
    }

    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {
      return delegate().execute(sql, columnNames);
    }

    @Override
    public int[] executeBatch() throws SQLException {
      return delegate().executeBatch();
    }

    @Override
    public ResultSet executeQuery(String sql) throws SQLException {
      return delegate().executeQuery(sql);
    }

    @Override
    public int executeUpdate(String sql) throws SQLException {
      return delegate().executeUpdate(sql);
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
      return delegate().executeUpdate(sql, autoGeneratedKeys);
    }

    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
      return delegate().executeUpdate(sql, columnIndexes);
    }

    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
      return delegate().executeUpdate(sql, columnNames);
    }

    @Override
    public Connection getConnection() throws SQLException {
      return delegate().getConnection();
    }

    @Override
    public int getFetchDirection() throws SQLException {
      return delegate().getFetchDirection();
    }

    @Override
    public int getFetchSize() throws SQLException {
      return delegate().getFetchSize();
    }

    @Override
    public ResultSet getGeneratedKeys() throws SQLException {
      return delegate().getGeneratedKeys();
    }

    @Override
    public int getMaxFieldSize() throws SQLException {
      return delegate().getMaxFieldSize();
    }

    @Override
    public int getMaxRows() throws SQLException {
      return delegate().getMaxRows();
    }

    @Override
    public boolean getMoreResults() throws SQLException {
      return delegate().getMoreResults();
    }

    @Override
    public boolean getMoreResults(int current) throws SQLException {
      return delegate().getMoreResults(current);
    }

    @Override
    public int getQueryTimeout() throws SQLException {
      return delegate().getQueryTimeout();
    }

    @Override
    public ResultSet getResultSet() throws SQLException {
      return delegate().getResultSet();
    }

    @Override
    public int getUpdateCount() throws SQLException {
      return delegate().getUpdateCount();
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
      return delegate().getWarnings();
    }

    @Override
    public boolean isCloseOnCompletion() throws SQLException {
      return delegate().isCloseOnCompletion();
    }

    @Override
    public boolean isClosed() throws SQLException {
      return delegate().isClosed();
    }

    @Override
    public boolean isPoolable() throws SQLException {
      return delegate().isPoolable();
    }

    @Override
    public void setCursorName(String name) throws SQLException {
      action((s) -> { s.setCursorName(name); return null; });
    }

    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException {
      action((s) -> { s.setEscapeProcessing(enable); return null; });
    }

    @Override
    public void setFetchDirection(int direction) throws SQLException {
      action((s) -> { s.setFetchDirection(direction); return null; });
    }

    @Override
    public void setFetchSize(int rows) throws SQLException {
      action((s) -> { s.setFetchSize(rows); return null; });
    }

    @Override
    public void setMaxFieldSize(int max) throws SQLException {
      action((s) -> { s.setMaxFieldSize(max); return null; });
    }

    @Override
    public void setMaxRows(int max) throws SQLException {
      action((s) -> { s.setMaxRows(max); return null; });
    }

    @Override
    public void setPoolable(boolean poolable) throws SQLException {
      action((s) -> { s.setPoolable(poolable); return null; });
    }

    @Override
    public void setQueryTimeout(int seconds) throws SQLException {
      action((s) -> { s.setQueryTimeout(seconds); return null; });
    }
  }
  
  /**
   * Class which attempts to delay generating a delegating {@link Statement} for as long as 
   * possible.  Queuing up actions so that once a delegating connection is selected we apply them 
   * as needed.
   */
  protected class DelegatingStatement extends AbstractDelegatingStatement<Statement> {
    public DelegatingStatement() throws SQLException {
      super();
    }

    public DelegatingStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
      super(resultSetType, resultSetConcurrency);
    }

    public DelegatingStatement(int resultSetType, int resultSetConcurrency,
                               int resultSetHoldability) throws SQLException {
      super(resultSetType, resultSetConcurrency, resultSetHoldability);
    }
    
    @Override
    protected SQLOperation<Connection, Statement> delegateStatementProvider() {
      if (resultSetHoldability == ResultSet.HOLD_CURSORS_OVER_COMMIT) {
        // minor optimization
        return (c) -> c.createStatement(resultSetType, resultSetConcurrency);
      } else {
        return (c) -> c.createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
      }
    }
  }

  /**
   * Class which attempts to delay generating a delegating {@link PreparedStatement} for as long as 
   * possible.  Queuing up actions so that once a delegating connection is selected we apply them 
   * as needed.
   */
  protected class DelegatingPreparedStatement extends AbstractDelegatingStatement<PreparedStatement>
                                              implements PreparedStatement {
    private final SQLOperation<Connection, PreparedStatement> delegateStatementProvider;
    
    public DelegatingPreparedStatement(String sql) throws SQLException {
      delegateStatementProvider = (c) -> c.prepareStatement(sql);
    }

    public DelegatingPreparedStatement(String sql, int[] columnIndexes) throws SQLException {
      delegateStatementProvider = (c) -> c.prepareStatement(sql, columnIndexes);
    }

    public DelegatingPreparedStatement(String sql, String[] columnNames) throws SQLException {
      delegateStatementProvider = (c) -> c.prepareStatement(sql, columnNames);
    }

    public DelegatingPreparedStatement(String sql, int autoGeneratedKeys) throws SQLException {
      delegateStatementProvider = (c) -> c.prepareStatement(sql, autoGeneratedKeys);
    }

    public DelegatingPreparedStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
      super(resultSetType, resultSetConcurrency);
      
      delegateStatementProvider = (c) -> c.prepareStatement(sql, resultSetType, resultSetConcurrency);
    }

    public DelegatingPreparedStatement(String sql, int resultSetType, int resultSetConcurrency,
                                       int resultSetHoldability) throws SQLException {
      super(resultSetType, resultSetConcurrency, resultSetHoldability);
      
      delegateStatementProvider = 
          (c) -> c.prepareStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
    }
    
    @Override
    protected SQLOperation<Connection, PreparedStatement> delegateStatementProvider() {
      return delegateStatementProvider;
    }

    @Override
    public void addBatch() throws SQLException {
      action((ps) -> { ps.addBatch(); return null; });
    }

    @Override
    public void clearParameters() throws SQLException {
      action((ps) -> { ps.clearParameters(); return null; });
    }

    @Override
    public boolean execute() throws SQLException {
      return delegate().execute();
    }

    @Override
    public ResultSet executeQuery() throws SQLException {
      return delegate().executeQuery();
    }

    @Override
    public int executeUpdate() throws SQLException {
      return delegate().executeUpdate();
    }

    @Override
    public ResultSetMetaData getMetaData() throws SQLException {
      return delegate().getMetaData();
    }

    @Override
    public ParameterMetaData getParameterMetaData() throws SQLException {
      return delegate().getParameterMetaData();
    }

    @Override
    public void setArray(int arg0, Array arg1) throws SQLException {
      action((ps) -> { ps.setArray(arg0, arg1); return null; });
    }

    @Override
    public void setAsciiStream(int arg0, InputStream arg1) throws SQLException {
      action((ps) -> { ps.setAsciiStream(arg0, arg1); return null; });
    }

    @Override
    public void setAsciiStream(int arg0, InputStream arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setAsciiStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setAsciiStream(int arg0, InputStream arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setAsciiStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setBigDecimal(int arg0, BigDecimal arg1) throws SQLException {
      action((ps) -> { ps.setBigDecimal(arg0, arg1); return null; });
    }

    @Override
    public void setBinaryStream(int arg0, InputStream arg1) throws SQLException {
      action((ps) -> { ps.setBinaryStream(arg0, arg1); return null; });
    }

    @Override
    public void setBinaryStream(int arg0, InputStream arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setBinaryStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setBinaryStream(int arg0, InputStream arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setBinaryStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setBlob(int arg0, Blob arg1) throws SQLException {
      action((ps) -> { ps.setBlob(arg0, arg1); return null; });
    }

    @Override
    public void setBlob(int arg0, InputStream arg1) throws SQLException {
      action((ps) -> { ps.setBlob(arg0, arg1); return null; });
    }

    @Override
    public void setBlob(int arg0, InputStream arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setBlob(arg0, arg1); return null; });
    }

    @Override
    public void setBoolean(int arg0, boolean arg1) throws SQLException {
      action((ps) -> { ps.setBoolean(arg0, arg1); return null; });
    }

    @Override
    public void setByte(int arg0, byte arg1) throws SQLException {
      action((ps) -> { ps.setByte(arg0, arg1); return null; });
    }

    @Override
    public void setBytes(int arg0, byte[] arg1) throws SQLException {
      action((ps) -> { ps.setBytes(arg0, arg1); return null; });
    }

    @Override
    public void setCharacterStream(int arg0, Reader arg1) throws SQLException {
      action((ps) -> { ps.setCharacterStream(arg0, arg1); return null; });
    }

    @Override
    public void setCharacterStream(int arg0, Reader arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setCharacterStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setCharacterStream(int arg0, Reader arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setCharacterStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setClob(int arg0, Clob arg1) throws SQLException {
      action((ps) -> { ps.setClob(arg0, arg1); return null; });
    }

    @Override
    public void setClob(int arg0, Reader arg1) throws SQLException {
      action((ps) -> { ps.setClob(arg0, arg1); return null; });
    }

    @Override
    public void setClob(int arg0, Reader arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setClob(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setDate(int arg0, Date arg1) throws SQLException {
      action((ps) -> { ps.setDate(arg0, arg1); return null; });
    }

    @Override
    public void setDate(int arg0, Date arg1, Calendar arg2) throws SQLException {
      action((ps) -> { ps.setDate(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setDouble(int arg0, double arg1) throws SQLException {
      action((ps) -> { ps.setDouble(arg0, arg1); return null; });
    }

    @Override
    public void setFloat(int arg0, float arg1) throws SQLException {
      action((ps) -> { ps.setFloat(arg0, arg1); return null; });
    }

    @Override
    public void setInt(int arg0, int arg1) throws SQLException {
      action((ps) -> { ps.setInt(arg0, arg1); return null; });
    }

    @Override
    public void setLong(int arg0, long arg1) throws SQLException {
      action((ps) -> { ps.setLong(arg0, arg1); return null; });
    }

    @Override
    public void setNCharacterStream(int arg0, Reader arg1) throws SQLException {
      action((ps) -> { ps.setNCharacterStream(arg0, arg1); return null; });
    }

    @Override
    public void setNCharacterStream(int arg0, Reader arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setNCharacterStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setNClob(int arg0, NClob arg1) throws SQLException {
      action((ps) -> { ps.setNClob(arg0, arg1); return null; });
    }

    @Override
    public void setNClob(int arg0, Reader arg1) throws SQLException {
      action((ps) -> { ps.setNClob(arg0, arg1); return null; });
    }

    @Override
    public void setNClob(int arg0, Reader arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setNClob(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setNString(int arg0, String arg1) throws SQLException {
      action((ps) -> { ps.setNString(arg0, arg1); return null; });
    }

    @Override
    public void setNull(int arg0, int arg1) throws SQLException {
      action((ps) -> { ps.setNull(arg0, arg1); return null; });
    }

    @Override
    public void setNull(int arg0, int arg1, String arg2) throws SQLException {
      action((ps) -> { ps.setNull(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setObject(int arg0, Object arg1) throws SQLException {
      action((ps) -> { ps.setObject(arg0, arg1); return null; });
    }

    @Override
    public void setObject(int arg0, Object arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setObject(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setObject(int arg0, Object arg1, int arg2, int arg3) throws SQLException {
      action((ps) -> { ps.setObject(arg0, arg1, arg2, arg3); return null; });
    }

    @Override
    public void setRef(int arg0, Ref arg1) throws SQLException {
      action((ps) -> { ps.setRef(arg0, arg1); return null; });
    }

    @Override
    public void setRowId(int arg0, RowId arg1) throws SQLException {
      action((ps) -> { ps.setRowId(arg0, arg1); return null; });
    }

    @Override
    public void setSQLXML(int arg0, SQLXML arg1) throws SQLException {
      action((ps) -> { ps.setSQLXML(arg0, arg1); return null; });
    }

    @Override
    public void setShort(int arg0, short arg1) throws SQLException {
      action((ps) -> { ps.setShort(arg0, arg1); return null; });
    }

    @Override
    public void setString(int arg0, String arg1) throws SQLException {
      action((ps) -> { ps.setString(arg0, arg1); return null; });
    }

    @Override
    public void setTime(int arg0, Time arg1) throws SQLException {
      action((ps) -> { ps.setTime(arg0, arg1); return null; });
    }

    @Override
    public void setTime(int arg0, Time arg1, Calendar arg2) throws SQLException {
      action((ps) -> { ps.setTime(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setTimestamp(int arg0, Timestamp arg1) throws SQLException {
      action((ps) -> { ps.setTimestamp(arg0, arg1); return null; });
    }

    @Override
    public void setTimestamp(int arg0, Timestamp arg1, Calendar arg2) throws SQLException {
      action((ps) -> { ps.setTimestamp(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setURL(int arg0, URL arg1) throws SQLException {
      action((ps) -> { ps.setURL(arg0, arg1); return null; });
    }

    @Override
    @SuppressWarnings("deprecation")
    public void setUnicodeStream(int arg0, InputStream arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setUnicodeStream(arg0, arg1, arg2); return null; });
    }
  }

  /**
   * Class which attempts to delay generating a delegating {@link CallableStatement} for as long as 
   * possible.  Queuing up actions so that once a delegating connection is selected we apply them 
   * as needed.
   */
  protected class DelegatingCallableStatement extends AbstractDelegatingStatement<CallableStatement> 
                                              implements CallableStatement {
    private final SQLOperation<Connection, CallableStatement> delegateStatementProvider;
    
    public DelegatingCallableStatement(String sql) throws SQLException {
      delegateStatementProvider = (c) -> c.prepareCall(sql);
    }

    public DelegatingCallableStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
      super(resultSetType, resultSetConcurrency);
      
      delegateStatementProvider = (c) -> c.prepareCall(sql, resultSetType, resultSetConcurrency);
    }

    public DelegatingCallableStatement(String sql, int resultSetType, int resultSetConcurrency,
                                       int resultSetHoldability) throws SQLException {
      super(resultSetType, resultSetConcurrency, resultSetHoldability);
      
      delegateStatementProvider = 
          (c) -> c.prepareCall(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
    }
    
    @Override
    protected SQLOperation<Connection, CallableStatement> delegateStatementProvider() {
      return delegateStatementProvider;
    }

    @Override
    public void addBatch() throws SQLException {
      action((ps) -> { ps.addBatch(); return null; });
    }

    @Override
    public void clearParameters() throws SQLException {
      action((ps) -> { ps.clearParameters(); return null; });
    }

    @Override
    public boolean execute() throws SQLException {
      return delegate().execute();
    }

    @Override
    public ResultSet executeQuery() throws SQLException {
      return delegate().executeQuery();
    }

    @Override
    public int executeUpdate() throws SQLException {
      return delegate().executeUpdate();
    }

    @Override
    public ResultSetMetaData getMetaData() throws SQLException {
      return delegate().getMetaData();
    }

    @Override
    public ParameterMetaData getParameterMetaData() throws SQLException {
      return delegate().getParameterMetaData();
    }

    @Override
    public void setArray(int arg0, Array arg1) throws SQLException {
      action((ps) -> { ps.setArray(arg0, arg1); return null; });
    }

    @Override
    public void setAsciiStream(int arg0, InputStream arg1) throws SQLException {
      action((ps) -> { ps.setAsciiStream(arg0, arg1); return null; });
    }

    @Override
    public void setAsciiStream(int arg0, InputStream arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setAsciiStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setAsciiStream(int arg0, InputStream arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setAsciiStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setBigDecimal(int arg0, BigDecimal arg1) throws SQLException {
      action((ps) -> { ps.setBigDecimal(arg0, arg1); return null; });
    }

    @Override
    public void setBinaryStream(int arg0, InputStream arg1) throws SQLException {
      action((ps) -> { ps.setBinaryStream(arg0, arg1); return null; });
    }

    @Override
    public void setBinaryStream(int arg0, InputStream arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setBinaryStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setBinaryStream(int arg0, InputStream arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setBinaryStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setBlob(int arg0, Blob arg1) throws SQLException {
      action((ps) -> { ps.setBlob(arg0, arg1); return null; });
    }

    @Override
    public void setBlob(int arg0, InputStream arg1) throws SQLException {
      action((ps) -> { ps.setBlob(arg0, arg1); return null; });
    }

    @Override
    public void setBlob(int arg0, InputStream arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setBlob(arg0, arg1); return null; });
    }

    @Override
    public void setBoolean(int arg0, boolean arg1) throws SQLException {
      action((ps) -> { ps.setBoolean(arg0, arg1); return null; });
    }

    @Override
    public void setByte(int arg0, byte arg1) throws SQLException {
      action((ps) -> { ps.setByte(arg0, arg1); return null; });
    }

    @Override
    public void setBytes(int arg0, byte[] arg1) throws SQLException {
      action((ps) -> { ps.setBytes(arg0, arg1); return null; });
    }

    @Override
    public void setCharacterStream(int arg0, Reader arg1) throws SQLException {
      action((ps) -> { ps.setCharacterStream(arg0, arg1); return null; });
    }

    @Override
    public void setCharacterStream(int arg0, Reader arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setCharacterStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setCharacterStream(int arg0, Reader arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setCharacterStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setClob(int arg0, Clob arg1) throws SQLException {
      action((ps) -> { ps.setClob(arg0, arg1); return null; });
    }

    @Override
    public void setClob(int arg0, Reader arg1) throws SQLException {
      action((ps) -> { ps.setClob(arg0, arg1); return null; });
    }

    @Override
    public void setClob(int arg0, Reader arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setClob(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setDate(int arg0, Date arg1) throws SQLException {
      action((ps) -> { ps.setDate(arg0, arg1); return null; });
    }

    @Override
    public void setDate(int arg0, Date arg1, Calendar arg2) throws SQLException {
      action((ps) -> { ps.setDate(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setDouble(int arg0, double arg1) throws SQLException {
      action((ps) -> { ps.setDouble(arg0, arg1); return null; });
    }

    @Override
    public void setFloat(int arg0, float arg1) throws SQLException {
      action((ps) -> { ps.setFloat(arg0, arg1); return null; });
    }

    @Override
    public void setInt(int arg0, int arg1) throws SQLException {
      action((ps) -> { ps.setInt(arg0, arg1); return null; });
    }

    @Override
    public void setLong(int arg0, long arg1) throws SQLException {
      action((ps) -> { ps.setLong(arg0, arg1); return null; });
    }

    @Override
    public void setNCharacterStream(int arg0, Reader arg1) throws SQLException {
      action((ps) -> { ps.setNCharacterStream(arg0, arg1); return null; });
    }

    @Override
    public void setNCharacterStream(int arg0, Reader arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setNCharacterStream(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setNClob(int arg0, NClob arg1) throws SQLException {
      action((ps) -> { ps.setNClob(arg0, arg1); return null; });
    }

    @Override
    public void setNClob(int arg0, Reader arg1) throws SQLException {
      action((ps) -> { ps.setNClob(arg0, arg1); return null; });
    }

    @Override
    public void setNClob(int arg0, Reader arg1, long arg2) throws SQLException {
      action((ps) -> { ps.setNClob(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setNString(int arg0, String arg1) throws SQLException {
      action((ps) -> { ps.setNString(arg0, arg1); return null; });
    }

    @Override
    public void setNull(int arg0, int arg1) throws SQLException {
      action((ps) -> { ps.setNull(arg0, arg1); return null; });
    }

    @Override
    public void setNull(int arg0, int arg1, String arg2) throws SQLException {
      action((ps) -> { ps.setNull(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setObject(int arg0, Object arg1) throws SQLException {
      action((ps) -> { ps.setObject(arg0, arg1); return null; });
    }

    @Override
    public void setObject(int arg0, Object arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setObject(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setObject(int arg0, Object arg1, int arg2, int arg3) throws SQLException {
      action((ps) -> { ps.setObject(arg0, arg1, arg2, arg3); return null; });
    }

    @Override
    public void setRef(int arg0, Ref arg1) throws SQLException {
      action((ps) -> { ps.setRef(arg0, arg1); return null; });
    }

    @Override
    public void setRowId(int arg0, RowId arg1) throws SQLException {
      action((ps) -> { ps.setRowId(arg0, arg1); return null; });
    }

    @Override
    public void setSQLXML(int arg0, SQLXML arg1) throws SQLException {
      action((ps) -> { ps.setSQLXML(arg0, arg1); return null; });
    }

    @Override
    public void setShort(int arg0, short arg1) throws SQLException {
      action((ps) -> { ps.setShort(arg0, arg1); return null; });
    }

    @Override
    public void setString(int arg0, String arg1) throws SQLException {
      action((ps) -> { ps.setString(arg0, arg1); return null; });
    }

    @Override
    public void setTime(int arg0, Time arg1) throws SQLException {
      action((ps) -> { ps.setTime(arg0, arg1); return null; });
    }

    @Override
    public void setTime(int arg0, Time arg1, Calendar arg2) throws SQLException {
      action((ps) -> { ps.setTime(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setTimestamp(int arg0, Timestamp arg1) throws SQLException {
      action((ps) -> { ps.setTimestamp(arg0, arg1); return null; });
    }

    @Override
    public void setTimestamp(int arg0, Timestamp arg1, Calendar arg2) throws SQLException {
      action((ps) -> { ps.setTimestamp(arg0, arg1, arg2); return null; });
    }

    @Override
    public void setURL(int arg0, URL arg1) throws SQLException {
      action((ps) -> { ps.setURL(arg0, arg1); return null; });
    }

    @Override
    @SuppressWarnings("deprecation")
    public void setUnicodeStream(int arg0, InputStream arg1, int arg2) throws SQLException {
      action((ps) -> { ps.setUnicodeStream(arg0, arg1, arg2); return null; });
    }
    
    @Override
    public Array getArray(int parameterIndex) throws SQLException {
      return delegate().getArray(parameterIndex);
    }

    @Override
    public Array getArray(String parameterName) throws SQLException {
      return delegate().getArray(parameterName);
    }

    @Override
    public BigDecimal getBigDecimal(int parameterIndex) throws SQLException {
      return delegate().getBigDecimal(parameterIndex);
    }

    @Override
    public BigDecimal getBigDecimal(String parameterName) throws SQLException {
      return delegate().getBigDecimal(parameterName);
    }

    @Override
    @SuppressWarnings("deprecation")
    public BigDecimal getBigDecimal(int parameterIndex, int scale) throws SQLException {
      return delegate().getBigDecimal(parameterIndex, scale);
    }

    @Override
    public Blob getBlob(int parameterIndex) throws SQLException {
      return delegate().getBlob(parameterIndex);
    }

    @Override
    public Blob getBlob(String parameterName) throws SQLException {
      return delegate().getBlob(parameterName);
    }

    @Override
    public boolean getBoolean(int parameterIndex) throws SQLException {
      return delegate().getBoolean(parameterIndex);
    }

    @Override
    public boolean getBoolean(String parameterName) throws SQLException {
      return delegate().getBoolean(parameterName);
    }

    @Override
    public byte getByte(int parameterIndex) throws SQLException {
      return delegate().getByte(parameterIndex);
    }

    @Override
    public byte getByte(String parameterName) throws SQLException {
      return delegate().getByte(parameterName);
    }

    @Override
    public byte[] getBytes(int parameterIndex) throws SQLException {
      return delegate().getBytes(parameterIndex);
    }

    @Override
    public byte[] getBytes(String parameterName) throws SQLException {
      return delegate().getBytes(parameterName);
    }

    @Override
    public Reader getCharacterStream(int parameterIndex) throws SQLException {
      return delegate().getCharacterStream(parameterIndex);
    }

    @Override
    public Reader getCharacterStream(String parameterName) throws SQLException {
      return delegate().getCharacterStream(parameterName);
    }

    @Override
    public Clob getClob(int parameterIndex) throws SQLException {
      return delegate().getClob(parameterIndex);
    }

    @Override
    public Clob getClob(String parameterName) throws SQLException {
      return delegate().getClob(parameterName);
    }

    @Override
    public Date getDate(int parameterIndex) throws SQLException {
      return delegate().getDate(parameterIndex);
    }

    @Override
    public Date getDate(String parameterName) throws SQLException {
      return delegate().getDate(parameterName);
    }

    @Override
    public Date getDate(int parameterIndex, Calendar cal) throws SQLException {
      return delegate().getDate(parameterIndex, cal);
    }

    @Override
    public Date getDate(String parameterName, Calendar cal) throws SQLException {
      return delegate().getDate(parameterName, cal);
    }

    @Override
    public double getDouble(int parameterIndex) throws SQLException {
      return delegate().getDouble(parameterIndex);
    }

    @Override
    public double getDouble(String parameterName) throws SQLException {
      return delegate().getDouble(parameterName);
    }

    @Override
    public float getFloat(int parameterIndex) throws SQLException {
      return delegate().getFloat(parameterIndex);
    }

    @Override
    public float getFloat(String parameterName) throws SQLException {
      return delegate().getFloat(parameterName);
    }

    @Override
    public int getInt(int parameterIndex) throws SQLException {
      return delegate().getInt(parameterIndex);
    }

    @Override
    public int getInt(String parameterName) throws SQLException {
      return delegate().getInt(parameterName);
    }

    @Override
    public long getLong(int parameterIndex) throws SQLException {
      return delegate().getLong(parameterIndex);
    }

    @Override
    public long getLong(String parameterName) throws SQLException {
      return delegate().getLong(parameterName);
    }

    @Override
    public Reader getNCharacterStream(int parameterIndex) throws SQLException {
      return delegate().getNCharacterStream(parameterIndex);
    }

    @Override
    public Reader getNCharacterStream(String parameterName) throws SQLException {
      return delegate().getNCharacterStream(parameterName);
    }

    @Override
    public NClob getNClob(int parameterIndex) throws SQLException {
      return delegate().getNClob(parameterIndex);
    }

    @Override
    public NClob getNClob(String parameterName) throws SQLException {
      return delegate().getNClob(parameterName);
    }

    @Override
    public String getNString(int parameterIndex) throws SQLException {
      return delegate().getNString(parameterIndex);
    }

    @Override
    public String getNString(String parameterName) throws SQLException {
      return delegate().getNString(parameterName);
    }

    @Override
    public Object getObject(int parameterIndex) throws SQLException {
      return delegate().getObject(parameterIndex);
    }

    @Override
    public Object getObject(String parameterName) throws SQLException {
      return delegate().getObject(parameterName);
    }

    @Override
    public Object getObject(int parameterIndex, Map<String, Class<?>> map) throws SQLException {
      return delegate().getObject(parameterIndex, map);
    }

    @Override
    public Object getObject(String parameterName, Map<String, Class<?>> map) throws SQLException {
      return delegate().getObject(parameterName, map);
    }

    @Override
    public <T> T getObject(int parameterIndex, Class<T> type) throws SQLException {
      return delegate().getObject(parameterIndex, type);
    }

    @Override
    public <T> T getObject(String parameterName, Class<T> type) throws SQLException {
      return delegate().getObject(parameterName, type);
    }

    @Override
    public Ref getRef(int parameterIndex) throws SQLException {
      return delegate().getRef(parameterIndex);
    }

    @Override
    public Ref getRef(String parameterName) throws SQLException {
      return delegate().getRef(parameterName);
    }

    @Override
    public RowId getRowId(int parameterIndex) throws SQLException {
      return delegate().getRowId(parameterIndex);
    }

    @Override
    public RowId getRowId(String parameterName) throws SQLException {
      return delegate().getRowId(parameterName);
    }

    @Override
    public SQLXML getSQLXML(int parameterIndex) throws SQLException {
      return delegate().getSQLXML(parameterIndex);
    }

    @Override
    public SQLXML getSQLXML(String parameterName) throws SQLException {
      return delegate().getSQLXML(parameterName);
    }

    @Override
    public short getShort(int parameterIndex) throws SQLException {
      return delegate().getShort(parameterIndex);
    }

    @Override
    public short getShort(String parameterName) throws SQLException {
      return delegate().getShort(parameterName);
    }

    @Override
    public String getString(int parameterIndex) throws SQLException {
      return delegate().getString(parameterIndex);
    }

    @Override
    public String getString(String parameterName) throws SQLException {
      return delegate().getString(parameterName);
    }

    @Override
    public Time getTime(int parameterIndex) throws SQLException {
      return delegate().getTime(parameterIndex);
    }

    @Override
    public Time getTime(String parameterName) throws SQLException {
      return delegate().getTime(parameterName);
    }

    @Override
    public Time getTime(int parameterIndex, Calendar cal) throws SQLException {
      return delegate().getTime(parameterIndex, cal);
    }

    @Override
    public Time getTime(String parameterName, Calendar cal) throws SQLException {
      return delegate().getTime(parameterName, cal);
    }

    @Override
    public Timestamp getTimestamp(int parameterIndex) throws SQLException {
      return delegate().getTimestamp(parameterIndex);
    }

    @Override
    public Timestamp getTimestamp(String parameterName) throws SQLException {
      return delegate().getTimestamp(parameterName);
    }

    @Override
    public Timestamp getTimestamp(int parameterIndex, Calendar cal) throws SQLException {
      return delegate().getTimestamp(parameterIndex, cal);
    }

    @Override
    public Timestamp getTimestamp(String parameterName, Calendar cal) throws SQLException {
      return delegate().getTimestamp(parameterName, cal);
    }

    @Override
    public URL getURL(int parameterIndex) throws SQLException {
      return delegate().getURL(parameterIndex);
    }

    @Override
    public URL getURL(String parameterName) throws SQLException {
      return delegate().getURL(parameterName);
    }

    @Override
    public boolean wasNull() throws SQLException {
      return delegate().wasNull();
    }

    @Override
    public void registerOutParameter(int parameterIndex, int sqlType) throws SQLException {
      action((cs) -> { cs.registerOutParameter(parameterIndex, sqlType); return null; });
    }

    @Override
    public void registerOutParameter(String parameterName, int sqlType) throws SQLException {
      action((cs) -> { cs.registerOutParameter(parameterName, sqlType); return null; });
    }

    @Override
    public void registerOutParameter(int parameterIndex, int sqlType, int scale) throws SQLException {
      action((cs) -> { cs.registerOutParameter(parameterIndex, sqlType, scale); return null; });
    }

    @Override
    public void registerOutParameter(int parameterIndex, int sqlType, String typeName) throws SQLException {
      action((cs) -> { cs.registerOutParameter(parameterIndex, sqlType, typeName); return null; });
    }

    @Override
    public void registerOutParameter(String parameterName, int sqlType, int scale) throws SQLException {
      action((cs) -> { cs.registerOutParameter(parameterName, sqlType, scale); return null; });
    }

    @Override
    public void registerOutParameter(String parameterName, int sqlType, String typeName) throws SQLException {
      action((cs) -> { cs.registerOutParameter(parameterName, sqlType, typeName); return null; });
    }

    @Override
    public void setAsciiStream(String parameterName, InputStream x) throws SQLException {
      action((cs) -> { cs.setAsciiStream(parameterName, x); return null; });
    }

    @Override
    public void setAsciiStream(String parameterName, InputStream x, int length) throws SQLException {
      action((cs) -> { cs.setAsciiStream(parameterName, x, length); return null; });
    }

    @Override
    public void setAsciiStream(String parameterName, InputStream x, long length) throws SQLException {
      action((cs) -> { cs.setAsciiStream(parameterName, x, length); return null; });
    }

    @Override
    public void setBigDecimal(String parameterName, BigDecimal x) throws SQLException {
      action((cs) -> { cs.setBigDecimal(parameterName, x); return null; });
    }

    @Override
    public void setBinaryStream(String parameterName, InputStream x) throws SQLException {
      action((cs) -> { cs.setBinaryStream(parameterName, x); return null; });
    }

    @Override
    public void setBinaryStream(String parameterName, InputStream x, int length) throws SQLException {
      action((cs) -> { cs.setBinaryStream(parameterName, x, length); return null; });
    }

    @Override
    public void setBinaryStream(String parameterName, InputStream x, long length) throws SQLException {
      action((cs) -> { cs.setBinaryStream(parameterName, x, length); return null; });
    }

    @Override
    public void setBlob(String parameterName, Blob x) throws SQLException {
      action((cs) -> { cs.setBlob(parameterName, x); return null; });
    }

    @Override
    public void setBlob(String parameterName, InputStream inputStream) throws SQLException {
      action((cs) -> { cs.setBlob(parameterName, inputStream); return null; });
    }

    @Override
    public void setBlob(String parameterName, InputStream inputStream, long length) throws SQLException {
      action((cs) -> { cs.setBlob(parameterName, inputStream, length); return null; });
    }

    @Override
    public void setBoolean(String parameterName, boolean x) throws SQLException {
      action((cs) -> { cs.setBoolean(parameterName, x); return null; });
    }

    @Override
    public void setByte(String parameterName, byte x) throws SQLException {
      action((cs) -> { cs.setByte(parameterName, x); return null; });
    }

    @Override
    public void setBytes(String parameterName, byte[] x) throws SQLException {
      action((cs) -> { cs.setBytes(parameterName, x); return null; });
    }

    @Override
    public void setCharacterStream(String parameterName, Reader reader) throws SQLException {
      action((cs) -> { cs.setCharacterStream(parameterName, reader); return null; });
    }

    @Override
    public void setCharacterStream(String parameterName, Reader reader, int length) throws SQLException {
      action((cs) -> { cs.setCharacterStream(parameterName, reader, length); return null; });
    }

    @Override
    public void setCharacterStream(String parameterName, Reader reader, long length) throws SQLException {
      action((cs) -> { cs.setCharacterStream(parameterName, reader, length); return null; });
    }

    @Override
    public void setClob(String parameterName, Clob x) throws SQLException {
      action((cs) -> { cs.setClob(parameterName, x); return null; });
    }

    @Override
    public void setClob(String parameterName, Reader reader) throws SQLException {
      action((cs) -> { cs.setClob(parameterName, reader); return null; });
    }

    @Override
    public void setClob(String parameterName, Reader reader, long length) throws SQLException {
      action((cs) -> { cs.setClob(parameterName, reader, length); return null; });
    }

    @Override
    public void setDate(String parameterName, Date x) throws SQLException {
      action((cs) -> { cs.setDate(parameterName, x); return null; });
    }

    @Override
    public void setDate(String parameterName, Date x, Calendar cal) throws SQLException {
      action((cs) -> { cs.setDate(parameterName, x, cal); return null; });
    }

    @Override
    public void setDouble(String parameterName, double x) throws SQLException {
      action((cs) -> { cs.setDouble(parameterName, x); return null; });
    }

    @Override
    public void setFloat(String parameterName, float x) throws SQLException {
      action((cs) -> { cs.setFloat(parameterName, x); return null; });
    }

    @Override
    public void setInt(String parameterName, int x) throws SQLException {
      action((cs) -> { cs.setInt(parameterName, x); return null; });
    }

    @Override
    public void setLong(String parameterName, long x) throws SQLException {
      action((cs) -> { cs.setLong(parameterName, x); return null; });
    }

    @Override
    public void setNCharacterStream(String parameterName, Reader value) throws SQLException {
      action((cs) -> { cs.setNCharacterStream(parameterName, value); return null; });
    }

    @Override
    public void setNCharacterStream(String parameterName, Reader value, long length) throws SQLException {
      action((cs) -> { cs.setNCharacterStream(parameterName, value, length); return null; });
    }

    @Override
    public void setNClob(String parameterName, NClob value) throws SQLException {
      action((cs) -> { cs.setNClob(parameterName, value); return null; });
    }

    @Override
    public void setNClob(String parameterName, Reader reader) throws SQLException {
      action((cs) -> { cs.setNClob(parameterName, reader); return null; });
    }

    @Override
    public void setNClob(String parameterName, Reader reader, long length) throws SQLException {
      action((cs) -> { cs.setNClob(parameterName, reader, length); return null; });
    }

    @Override
    public void setNString(String parameterName, String value) throws SQLException {
      action((cs) -> { cs.setNString(parameterName, value); return null; });
    }

    @Override
    public void setNull(String parameterName, int sqlType) throws SQLException {
      action((cs) -> { cs.setNull(parameterName, sqlType); return null; });
    }

    @Override
    public void setNull(String parameterName, int sqlType, String typeName) throws SQLException {
      action((cs) -> { cs.setNull(parameterName, sqlType, typeName); return null; });
    }

    @Override
    public void setObject(String parameterName, Object x) throws SQLException {
      action((cs) -> { cs.setObject(parameterName, x); return null; });
    }

    @Override
    public void setObject(String parameterName, Object x, int targetSqlType) throws SQLException {
      action((cs) -> { cs.setObject(parameterName, x, targetSqlType); return null; });
    }

    @Override
    public void setObject(String parameterName, Object x, int targetSqlType, int scale) throws SQLException {
      action((cs) -> { cs.setObject(parameterName, x, targetSqlType, scale); return null; });
    }

    @Override
    public void setRowId(String parameterName, RowId x) throws SQLException {
      action((cs) -> { cs.setRowId(parameterName, x); return null; });
    }

    @Override
    public void setSQLXML(String parameterName, SQLXML xmlObject) throws SQLException {
      action((cs) -> { cs.setSQLXML(parameterName, xmlObject); return null; });
    }

    @Override
    public void setShort(String parameterName, short x) throws SQLException {
      action((cs) -> { cs.setShort(parameterName, x); return null; });
    }

    @Override
    public void setString(String parameterName, String x) throws SQLException {
      action((cs) -> { cs.setString(parameterName, x); return null; });
    }

    @Override
    public void setTime(String parameterName, Time x) throws SQLException {
      action((cs) -> { cs.setTime(parameterName, x); return null; });
    }

    @Override
    public void setTime(String parameterName, Time x, Calendar cal) throws SQLException {
      action((cs) -> { cs.setTime(parameterName, x, cal); return null; });
    }

    @Override
    public void setTimestamp(String parameterName, Timestamp x) throws SQLException {
      action((cs) -> { cs.setTimestamp(parameterName, x); return null; });
    }

    @Override
    public void setTimestamp(String parameterName, Timestamp x, Calendar cal) throws SQLException {
      action((cs) -> { cs.setTimestamp(parameterName, x, cal); return null; });
    }

    @Override
    public void setURL(String parameterName, URL val) throws SQLException {
      action((cs) -> { cs.setURL(parameterName, val); return null; });
    }
  }
}
