package org.thryft.waf.cli;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.File;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.thryft.waf.lib.logging.LoggingConfigurator;

import com.beust.jcommander.IStringConverter;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;

import ch.qos.logback.core.joran.spi.JoranException;

public abstract class CliMain {
    public static class Args {
        @Parameter(names = { "-h", "--help" }, help = true)
        private boolean help;

        @Parameter(names = { "-ln",
                "--logback-xml-file-name" }, description = "name of logback.xml in /etc/notablist or ~/.notablist")
        public String logbackXmlFileName = null;

        @Parameter(names = { "-lp",
                "--logback-xml-file-path" }, converter = FileParameterConverter.class, description = "path to logback.xml")
        public File logbackXmlFilePath = null;
    }

    public final static class FileParameterConverter implements IStringConverter<File> {
        @Override
        public File convert(final String value) {
            final File file = new File(value);
            if (!file.isFile()) {
                throw new ParameterException(String.format("'%s' is not a valid file path", value));
            }
            return file;
        }
    }

    protected CliMain(final Marker logMarker) {
        this.logger = LoggerFactory.getLogger(getClass());
        this.logMarker = checkNotNull(logMarker);
    }

    protected final void _configureLogging(final Args args, final String projectName) throws JoranException {
        if (args.logbackXmlFileName != null) {
            LoggingConfigurator.configureLoggingFromFileName(args.logbackXmlFileName, getClass(), projectName);
        } else if (args.logbackXmlFilePath != null) {
            LoggingConfigurator.configureLoggingFromFilePath(args.logbackXmlFilePath);
        } else {
            throw new IllegalArgumentException("must specify --logback-xml-file-name or --logback-xml-file-path");
        }
    }

    protected final Logger _getLogger() {
        return logger;
    }

    protected final Marker _getLogMarker() {
        return logMarker;
    }

    private final Logger logger;
    private final Marker logMarker;
}
