package org.thryft.waf.server.controllers.oauth;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;

import java.io.IOException;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.thryft.native_.EmailAddress;
import org.thryft.waf.lib.logging.LoggingUtils;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.scribejava.apis.GoogleApi20;
import com.github.scribejava.core.model.OAuth2AccessToken;
import com.github.scribejava.core.model.OAuthRequest;
import com.github.scribejava.core.model.Response;
import com.github.scribejava.core.model.Verb;
import com.github.scribejava.core.oauth.OAuth20Service;
import com.google.common.base.Optional;

public final class GoogleOauth2ServiceProvider extends GoogleApi20 implements Oauth2ServiceProvider {
    public GoogleOauth2ServiceProvider(final String apiKey, final String apiSecret) {
        this.apiKey = checkNotNull(apiKey);
        this.apiSecret = checkNotNull(apiSecret);
    }

    @Override
    public String getApiKey() {
        return apiKey;
    }

    @Override
    public String getApiSecret() {
        return apiSecret;
    }

    @Override
    public Optional<String> getScope() {
        return SCOPE;
    }

    @Override
    public OauthUserProfile getUserProfile(final OAuth20Service service, final OAuth2AccessToken accessToken)
            throws IOException {
        final OAuthRequest request = new OAuthRequest(Verb.GET,
                "https://www.googleapis.com/oauth2/v1/userinfo?alt=json", service);
        service.signRequest(accessToken, request);
        final Response response = request.send();
        if (response.getCode() != 200) {
            throw new IOException("non-200 response code: " + response.getCode());
        }
        final OauthUserProfile.Builder resultBuilder = OauthUserProfile.builder();
        final Map<String, Object> userInfo = objectMapper.readValue(response.getBody(),
                new TypeReference<Map<String, Object>>() {
                });
        for (final Map.Entry<String, Object> entry : userInfo.entrySet()) {
            if (entry.getKey().equals("id")) {
                final String id = entry.getValue().toString();
                checkState(!id.isEmpty());
                resultBuilder.setUsername(id);
            } else if (entry.getKey().equals("email")) {
                EmailAddress email;
                try {
                    email = new EmailAddress((String) entry.getValue());
                } catch (ClassCastException | IllegalArgumentException e) {
                    logger.warn(logMarker, "error parsing {}: {}", entry.getKey(), entry.getValue());
                    continue;
                }
                resultBuilder.setEmailAddress(email);
                // } else if (entry.getValue().equals("family_name")) {
                // final String familyName = (String) entry.getValue();
                // if (!familyName.isEmpty()) {
                // userBuilder.setLastName(familyName);
                // }
                // } else if (entry.getKey().equals("given_name")) {
                // final String givenName = (String) entry.getValue();
                // if (!givenName.isEmpty()) {
                // userBuilder.setFirstName(givenName);
                // }
                // } else if (entry.getKey().equals("name")) {
                // final String name = (String) entry.getValue();
                // if (!name.isEmpty()) {
                // userBuilder.setDisplayName(name);
                // }
                // } else if (entry.getKey().equals("picture")) {
                // Url picture;
                // try {
                // picture = Url.parse((String) entry.getValue());
                // } catch (ClassCastException | IllegalArgumentException e) {
                // logger.warn("error parsing {}: {}", entry.getKey(),
                // entry.getValue());
                // continue;
                // }
                // userBuilder.setImageUrl(picture);
            } else {
                logger.debug(logMarker, "unknown userinfo field {}: {}", entry.getKey(), entry.getValue());
            }
        }
        return resultBuilder.build();
    }

    private final String apiKey;
    private final String apiSecret;

    private final ObjectMapper objectMapper = new ObjectMapper();

    public final static String ID = "google";

    private final static Logger logger = LoggerFactory.getLogger(GoogleOauth2ServiceProvider.class);
    private final static Marker logMarker = LoggingUtils.getMarker(GoogleOauth2ServiceProvider.class);

    private static final Optional<String> SCOPE = Optional.of("https://www.googleapis.com/auth/userinfo.email");
}
