package org.thryft.waf.server.controllers.oauth;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;

import java.io.IOException;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.thryft.native_.Url;
import org.thryft.waf.lib.logging.LoggingUtils;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.scribejava.core.model.OAuth2AccessToken;
import com.github.scribejava.core.model.OAuthRequest;
import com.github.scribejava.core.model.Response;
import com.github.scribejava.core.model.Verb;
import com.github.scribejava.core.oauth.OAuth20Service;
import com.google.common.base.Optional;

public final class FacebookOauth2ServiceProvider extends com.github.scribejava.apis.FacebookApi
implements Oauth2ServiceProvider {
    public FacebookOauth2ServiceProvider(final String apiKey, final String apiSecret) {
        this.apiKey = checkNotNull(apiKey);
        this.apiSecret = checkNotNull(apiSecret);
    }

    @Override
    public String getApiKey() {
        return apiKey;
    }

    @Override
    public String getApiSecret() {
        return apiSecret;
    }

    @Override
    public Optional<String> getScope() {
        return SCOPE;
    }

    @Override
    public OauthUserProfile getUserProfile(final OAuth20Service service, final OAuth2AccessToken accessToken)
            throws IncompleteOAuthUserProfileException, IOException {
        // Request fields in public_profile:

        final OAuthRequest request = new OAuthRequest(Verb.GET,
                "https://graph.facebook.com/me?fields=email," + PUBLIC_PROFILE_FIELDS,
                service);
        service.signRequest(accessToken, request);
        final Response response = request.send();
        if (response.getCode() != 200) {
            throw new IOException("non-200 response code: " + response.getCode());
        }
        final Map<String, Object> me = objectMapper.readValue(response.getBody(),
                new TypeReference<Map<String, Object>>() {
        });
        final OauthUserProfile.Builder resultBuilder = OauthUserProfile.builder();
        for (final Map.Entry<String, Object> entry : me.entrySet()) {
            switch (entry.getKey()) {
            case "email":
                resultBuilder.setEmailAddress((String) entry.getValue());
                break;
            case "first_name":
                final String firstName = (String) entry.getValue();
                if (!StringUtils.isBlank(firstName)) {
                    resultBuilder.setFirstName(Optional.of(firstName));
                }
                break;
            case "id":
                final String id = entry.getValue().toString();
                checkState(!StringUtils.isBlank(id));
                resultBuilder.setId(id);
                break;
            case "last_name":
                final String lastName = (String) entry.getValue();
                if (!StringUtils.isBlank(lastName)) {
                    resultBuilder.setLastName(Optional.of(lastName));
                }
                break;
            case "name":
                final String name = (String) entry.getValue();
                if (!StringUtils.isBlank(name)) {
                    resultBuilder.setName(Optional.of(name));
                }
                break;
            case "picture":
                if (!(entry.getValue() instanceof Map)) {
                    continue;
                }
                @SuppressWarnings("unchecked")
                final Map<String, Object> picture = (Map<String, Object>) entry.getValue();
                final Object pictureData = picture.get("data");
                if (!(pictureData instanceof Map)) {
                    continue;
                }
                @SuppressWarnings("unchecked")
                final Object pictureUrl = ((Map<String, Object>) pictureData).get("url");
                if (!(pictureUrl instanceof String)) {
                    continue;
                }
                try {
                    resultBuilder.setPicture(Optional.of(Url.parse((String) pictureUrl)));
                } catch (final IllegalArgumentException e) {
                    logger.warn("error parsing {}: {}", entry.getKey(), entry.getValue());
                }
                break;
            default:
                logger.debug(logMarker, "unknown me field {}: {}", entry.getKey(), entry.getValue());
                break;
            }
        }
        try {
            return resultBuilder.build();
        } catch (final NullPointerException e) {
            throw new IncompleteOAuthUserProfileException(e);
        }
    }

    private final String apiKey;

    private final String apiSecret;
    private final ObjectMapper objectMapper = new ObjectMapper();

    private final static String PUBLIC_PROFILE_FIELDS = StringUtils.join(new String[]{
            "id",
            "cover",
            "name",
            "first_name",
            "last_name",
            "age_range",
            "link",
            "gender",
            "locale",
            "picture",
            "timezone",
            "updated_time",
    "verified"}, ',');

    public final static String ID = "facebook";

    private final static Logger logger = LoggerFactory.getLogger(FacebookOauth2ServiceProvider.class);
    private final static Marker logMarker = LoggingUtils.getMarker(FacebookOauth2ServiceProvider.class);

    private static final Optional<String> SCOPE = Optional.of("email");
}
