package org.thryft.waf.server;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.File;
import java.io.IOException;

import javax.annotation.Nullable;
import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;

import org.apache.shiro.mgt.SecurityManager;
import org.apache.shiro.web.mgt.CookieRememberMeManager;
import org.apache.shiro.web.mgt.DefaultWebSecurityManager;
import org.thryft.waf.lib.logging.LoggingConfigurator;

import com.google.inject.servlet.GuiceServletContextListener;

public abstract class AbstractServletContextListener extends GuiceServletContextListener {
    protected AbstractServletContextListener(final String projectName, final byte[] rememberMeCipherKey) {
        this.projectName = checkNotNull(projectName);
        this.rememberMeCipherKey = checkNotNull(rememberMeCipherKey);
    }

    @Override
    public void contextInitialized(final ServletContextEvent servletContextEvent) {
        __configureLogging(servletContextEvent.getServletContext());

        super.contextInitialized(servletContextEvent);

        ((CookieRememberMeManager) ((DefaultWebSecurityManager) getInjector().getInstance(SecurityManager.class))
                .getRememberMeManager()).setCipherKey(rememberMeCipherKey);
    }

    private void __configureLogging(final ServletContext servletContext) {
        {
            @Nullable
            final String logbackXmlFilePath = servletContext.getInitParameter("logbackXmlFilePath");
            if (logbackXmlFilePath != null) {
                try {
                    LoggingConfigurator.configureLoggingFromFilePath(new File(logbackXmlFilePath));
                    return;
                } catch (final IOException e) {
                }
            }
        }

        {
            @Nullable
            final String logbackXmlFileName = servletContext.getInitParameter("logbackXmlFileName");
            if (logbackXmlFileName != null) {
                try {
                    LoggingConfigurator.configureLoggingFromFileName(logbackXmlFileName, getClass(), projectName);
                    return;
                } catch (final IOException e) {
                }
            }
        }

        {
            try {
                LoggingConfigurator.configureLoggingFromFileName("logback.xml", getClass(), projectName);
            } catch (final IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private final String projectName;
    private final byte[] rememberMeCipherKey;
}
