package com.mdsol.rave;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.reactive.function.client.ClientRequest;
import org.springframework.web.reactive.function.client.ExchangeFilterFunction;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClient.ResponseSpec;
import org.springframework.web.util.UriBuilder;

import reactor.core.publisher.Mono;

public class RaveOdmClient {

	protected static Logger LOGGER = LoggerFactory.getLogger(RaveOdmClient.class);
	
	protected static final String METADATA_VERSION_URI = "RaveWebServices/metadata/studies/{study}/versions/{oid}";
	protected static final String METADATA_MATRIX_URI = "RaveWebServices/datasets/VersionFolders.odm";
	protected static final String METADATA_VERSIONS_URI = "RaveWebServices/metadata/studies/{study}/versions";
	protected static final String SIGNATURES_URI = "RaveWebServices/datasets/Signatures.odm";
	protected static final String SITES_URI = "RaveWebServices/datasets/Sites.odm";
	protected static final String STUDIES_URI = "RaveWebServices/studies/";
	protected static final String CLINICAL_RECORDS_URI = "RaveWebServices/datasets/ClinicalAuditRecords.odm";

	public static final String XSLT_RESOURCE = "transform-mdsol.xsl";

	// msic url strings
	protected static String linkHeader = "Link";
	protected static String studyOidParameter = "studyoid";
	protected static String studyIdParameter = "studyid";
	protected static String startIdParameter = "startid";
	protected static String locationOidParameter = "LocationOID";
	protected static String pageSizeParameter = "per_page";
	protected static String startIdRegex = "startid=([0-9]+)";
	protected static String studyOidEnvironmentRegex = "\\(.*?\\)\\z";
	protected static Pattern startIdPattern = Pattern.compile(startIdRegex);

	protected Boolean ssl = true;
	protected Transformer transformer;
	protected String encodedCredentials;
	protected String host;
	protected WebClient webClient;

	public RaveOdmClient() {}

	public void setBasicAuth(String user, String pass) {
		if (user != null || pass != null) {
			 encodedCredentials = HttpHeaders.encodeBasicAuth(user, pass, null);
		}
		else encodedCredentials = null;

	}
	
	public WebClient getClient() {

		if (webClient == null) {
			WebClient.Builder builder = WebClient.builder();
			ExchangeFilterFunction func = (request, next) -> {
			 		if(encodedCredentials == null) {
			 			return next.exchange(request);
			 		}
					return next.exchange(ClientRequest.from(request)
						.headers(headers -> headers.setBasicAuth(encodedCredentials))
						.build()
					);
			}
			;
			builder = builder.filter(func);
			
			webClient = builder.codecs(config -> {
				config.defaultCodecs().maxInMemorySize(1024 * 1024 * 10);
			}).build();
		}
		return webClient;
	}

	/*protected UriBuilder buildUri(String host,String path){
		UriBuilder ub = builderFactory.builder();
		ub.scheme(ssl?"https":"http");
		ub.host(host);
		ub.path(path);
		return ub;
	}*/

	protected UriBuilder setupUriBuilder(UriBuilder builder) {
		builder.scheme(ssl ? "https" : "http");
		builder.host(host);
		return builder;
	}

	public ResponseEntity<DataBuffer> getResponseEntity(ResponseSpec spec) {

		Mono<ResponseEntity<DataBuffer>> mono = spec.onStatus(HttpStatus::isError, response -> {
			if (response.statusCode() == HttpStatus.UNAUTHORIZED) {
				return Mono.error(new RaveClientUnauthorizedException(response.statusCode().toString()));
			} else
				return Mono.error(new RaveClientException(response.statusCode().toString()));
		})

		.toEntity(DataBuffer.class);

		ResponseEntity<DataBuffer> db = mono.block();

		return db;
	}

	protected RaveResponse handleResponseSpec(ResponseSpec spec, ResponseHandler handler) {
		Long startTime = System.currentTimeMillis();
		ResponseEntity<DataBuffer> response = getResponseEntity(spec);
		RaveResponse r = new RaveResponse(response, startTime);
		if(handler != null) handler.handle(r);
		return r;
	}

	public String getHost() {
		return host;
	}

	public void setHost(String host) {
		this.host = host;
	}

	public RaveResponse getClinicalData(String study, Long startId, Integer pageSize, ResponseHandler handler) {
		LOGGER.info("getClinicalData: {} , {} -> {}", new Object[] { study, startId, pageSize });

		ResponseSpec spec = getClient().get().uri(urib -> {
			return setupUriBuilder(urib).path(CLINICAL_RECORDS_URI)
					.queryParam(studyOidParameter, study)
					.queryParam(startIdParameter, (startId != null) ? startId.toString() : 0)
					.queryParam(pageSizeParameter, pageSize.toString()).build();
		}).retrieve();

		return handleResponseSpec(spec, handler);
	}

	public RaveResponse getStudies(ResponseHandler handler) {

		ResponseSpec spec = getClient().get().uri(urib -> {
			return setupUriBuilder(urib).path(STUDIES_URI).build();
		}).retrieve();

		return handleResponseSpec(spec, handler);

	}

	public RaveResponse getSites(String studyOid, ResponseHandler handler) {

		ResponseSpec spec = getClient().get().uri(urib -> {
			URI uri = setupUriBuilder(urib).path(SITES_URI)
					.queryParam(studyOidParameter, studyOid).build();
			return uri;
		}).retrieve();

		return handleResponseSpec(spec, handler);

	}

	public RaveResponse getSignatures(String studyOid, ResponseHandler handler) {
		//final String studyOidNoEnv = removeStudyEnvironment(studyOid);
		ResponseSpec spec = getClient().get().uri(urib -> {
			return setupUriBuilder(urib)
					.path(SIGNATURES_URI)
					.queryParam(studyIdParameter, studyOid)
					.build();
		}).retrieve();

		return handleResponseSpec(spec, handler);

	}

	public RaveResponse getMetaDataListOdm(String study, ResponseHandler handler) {

		final String studyOidNoEnv = removeStudyEnvironment(study);
		ResponseSpec spec = getClient().get().uri(urib -> {
			return setupUriBuilder(urib)
					.path(METADATA_VERSIONS_URI)
					.build(studyOidNoEnv);
		}).retrieve();

		return handleResponseSpec(spec, handler);

	}

	public RaveResponse getMetaDataMatricesOdm(String studyOid, ResponseHandler handler) {

		ResponseSpec spec = getClient().get().uri(urib -> {
			return setupUriBuilder(urib).path(METADATA_MATRIX_URI)
					.queryParam(studyOidParameter, studyOid).build();
		}).retrieve();

		return handleResponseSpec(spec, handler);

	}

	public RaveResponse getMetaDataVersionOdm(String study, String oid, ResponseHandler handler) {

		final String studyOidNoEnv = removeStudyEnvironment(study);
		ResponseSpec spec = getClient().get().uri(urib -> {
			return setupUriBuilder(urib).path(METADATA_VERSION_URI)
					.build(studyOidNoEnv, oid);
		}).retrieve();

		return handleResponseSpec(spec, handler);

	}


	public static interface ResponseHandler {
		public void handle(RaveResponse response);
	}

	public class RaveResponse {

		protected ResponseEntity<DataBuffer> response;
		protected Long start;

		public RaveResponse(ResponseEntity<DataBuffer> r, Long start) {
			this.response = r;
			this.start = start;
		}

		public InputStream getStream() {
			return response.getBody().asInputStream();
		}

		public Long getNextPage() {
			Long next = null;
			if (response.getHeaders().containsKey(linkHeader)) {
				String link = response.getHeaders().get(linkHeader).get(0);
				Matcher m = startIdPattern.matcher(link);
				if (m.find())
					next = Long.valueOf(m.group(1));
			}
			return next;
		}

		public Long getStart() {
			return start;
		}

	}

	public static String removeStudyEnvironment(String study) {
		return study.replaceAll(studyOidEnvironmentRegex, StringUtils.EMPTY);
	}

	public static String makeStudyParameter(String study, String environment) {
		StringBuilder sb = new StringBuilder(study);
		sb.append("(");
		sb.append(environment);
		sb.append(")");
		return sb.toString();
	}

	public void transform(InputStream input, OutputStream output) throws TransformerException {
		StreamSource sourcestream = new StreamSource(input);
		Transformer transformer = getTransformer();
		transformer.transform(sourcestream, new StreamResult(output));
	}

	public InputStream transform(InputStream input) {
		try {
			StreamSource sourcestream = new StreamSource(input);
			Transformer transformer = getTransformer();
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			transformer.transform(sourcestream, new StreamResult(baos));
			return new ByteArrayInputStream(baos.toByteArray());
		} catch (TransformerException e) {
			throw new RaveClientException(e);
		}
	}

	public Transformer getTransformer() {
		if (transformer == null) {
			try {
				TransformerFactory tf = new net.sf.saxon.TransformerFactoryImpl();
				StreamSource stylesource = new StreamSource(
						RaveOdmClient.class.getResourceAsStream(XSLT_RESOURCE));
				transformer = tf.newTransformer(stylesource);
			} catch (TransformerConfigurationException e) {
				throw new IllegalStateException(e);
			}
		}
		return transformer;
	}

}
