/*
 * Decompiled with CFR 0.152.
 */
package org.tinfour.gis.shapefile;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import org.tinfour.gis.shapefile.DbfFileReader;
import org.tinfour.gis.shapefile.ShapefileRecord;
import org.tinfour.gis.shapefile.ShapefileType;
import org.tinfour.io.BufferedRandomAccessReader;

public class ShapefileReader
implements Closeable {
    private final File file;
    private final BufferedRandomAccessReader raf;
    final int fileLength;
    final long fileLengthInBytes;
    final int version;
    ShapefileType shapefileType;
    private final double minX;
    private final double maxX;
    private final double minY;
    private final double maxY;
    private double minZ;
    private double maxZ;
    private int nPointsTotal;
    private int nPartsTotal;
    private int[] recordIndex;
    private int nRecordsInFile;

    public ShapefileReader(File file) throws IOException {
        this.file = file;
        this.raf = new BufferedRandomAccessReader(file);
        int fileCode = this.raf.readIntBigEndian();
        if (fileCode != 9994) {
            throw new IOException("Specified file is not a Shapefile " + file.getPath());
        }
        this.raf.seek(24L);
        this.fileLength = this.raf.readIntBigEndian();
        this.fileLengthInBytes = (long)this.fileLength * 2L;
        this.version = this.raf.readInt();
        int shapeTypeCode = this.raf.readInt();
        this.shapefileType = ShapefileType.getShapefileType(shapeTypeCode);
        if (this.shapefileType == null) {
            throw new IOException("Invalid Shapefile type code " + shapeTypeCode);
        }
        this.minX = this.raf.readDouble();
        this.minY = this.raf.readDouble();
        this.maxX = this.raf.readDouble();
        this.maxY = this.raf.readDouble();
        this.minZ = this.raf.readDouble();
        this.maxZ = this.raf.readDouble();
        this.raf.skipBytes(16);
        if (this.minZ == this.maxZ && this.minZ == 0.0) {
            this.minZ = Double.POSITIVE_INFINITY;
            this.maxZ = Double.NEGATIVE_INFINITY;
        }
    }

    private void readShxFile() throws IOException {
        if (this.recordIndex != null) {
            return;
        }
        File shxFile = this.getCoFile("shx");
        try (FileInputStream fins = new FileInputStream(shxFile);
             BufferedInputStream bins = new BufferedInputStream(fins);
             DataInputStream dins = new DataInputStream(bins);){
            int fileCode = dins.readInt();
            if (fileCode != 9994) {
                throw new IOException("Specified file is not a Shapefile " + this.file.getPath());
            }
            dins.skipBytes(20);
            int shxFileLength = dins.readInt();
            int shxFileLengthInBytes = shxFileLength * 2;
            dins.skip(72L);
            this.nRecordsInFile = (shxFileLengthInBytes - 100) / 8;
            this.recordIndex = new int[this.nRecordsInFile + 1];
            for (int i = 1; i <= this.nRecordsInFile; ++i) {
                this.recordIndex[i] = dins.readInt() * 2;
                dins.skip(4L);
            }
        }
    }

    @Override
    public void close() throws IOException {
        this.raf.close();
    }

    public double getMinX() {
        return this.minX;
    }

    public double getMaxX() {
        return this.maxX;
    }

    public double getMinY() {
        return this.minY;
    }

    public double getMaxY() {
        return this.maxY;
    }

    public double getMinZ() {
        return this.minZ;
    }

    public double getMaxZ() {
        return this.maxZ;
    }

    public ShapefileRecord readNextRecord(ShapefileRecord pRecord) throws IOException {
        ShapefileRecord record = pRecord;
        if (record == null) {
            record = new ShapefileRecord(this.shapefileType);
        } else {
            record.shapefileType = this.shapefileType;
        }
        long offset0 = this.raf.getFilePosition();
        if (offset0 >= this.fileLengthInBytes) {
            return null;
        }
        int recNo = this.raf.readIntBigEndian();
        int recLen = this.raf.readIntBigEndian();
        record.recordNumber = recNo;
        record.offset = offset0;
        int stc = this.raf.readInt();
        if (stc != this.shapefileType.getTypeCode()) {
            throw new IOException("Error reading Shapefile record, typecode mismatch, found " + stc + ", expected " + this.shapefileType.getTypeCode());
        }
        switch (this.shapefileType) {
            case Point: {
                record.setSizes(1, 1);
                record.nParts = 1;
                record.nPoints = 1;
                record.partStart[1] = 1;
                record.x0 = this.raf.readDouble();
                record.y0 = this.raf.readDouble();
                record.x1 = record.x0;
                record.y1 = record.y0;
                record.xyz[0] = record.x0;
                record.xyz[1] = record.y0;
                break;
            }
            case PointZ: {
                record.setSizes(1, 1);
                record.nParts = 1;
                record.nPoints = 1;
                record.partStart[1] = 1;
                record.x0 = this.raf.readDouble();
                record.y0 = this.raf.readDouble();
                record.z0 = this.raf.readDouble();
                record.x1 = record.x0;
                record.y1 = record.y0;
                record.z1 = record.z0;
                record.xyz[0] = record.x0;
                record.xyz[1] = record.y0;
                record.xyz[2] = record.z0;
                break;
            }
            case PolyLineZ: 
            case PolygonZ: {
                record.x0 = this.raf.readDouble();
                record.y0 = this.raf.readDouble();
                record.x1 = this.raf.readDouble();
                record.y1 = this.raf.readDouble();
                int nParts = this.raf.readInt();
                int nPoints = this.raf.readInt();
                record.setSizes(nPoints, nParts);
                int[] partStart = record.partStart;
                double[] xyz = record.xyz;
                this.nPointsTotal += nPoints;
                this.nPartsTotal += nParts;
                for (int iPart = 0; iPart < nParts; ++iPart) {
                    partStart[iPart] = this.raf.readInt();
                }
                partStart[nParts] = nPoints;
                int k = 0;
                for (int i = 0; i < nPoints; ++i) {
                    xyz[k++] = this.raf.readDouble();
                    xyz[k++] = this.raf.readDouble();
                    ++k;
                }
                record.z0 = this.raf.readDouble();
                record.z1 = this.raf.readDouble();
                if (record.z0 < this.minZ) {
                    this.minZ = record.z0;
                }
                if (record.z1 > this.maxZ) {
                    this.maxZ = record.z1;
                }
                for (int iPart = 0; iPart < nParts; ++iPart) {
                    int n = partStart[iPart + 1] - partStart[iPart];
                    for (int i = 0; i < n; ++i) {
                        xyz[i * 3 + 2] = this.raf.readDouble();
                    }
                }
                break;
            }
            case PolyLine: 
            case Polygon: {
                record.x0 = this.raf.readDouble();
                record.y0 = this.raf.readDouble();
                record.x1 = this.raf.readDouble();
                record.y1 = this.raf.readDouble();
                int nParts = this.raf.readInt();
                int nPoints = this.raf.readInt();
                record.setSizes(nPoints, nParts);
                this.nPointsTotal += nPoints;
                this.nPartsTotal += nParts;
                for (int iPart = 0; iPart < nParts; ++iPart) {
                    record.partStart[iPart] = this.raf.readInt();
                }
                record.partStart[nParts] = nPoints;
                int k = 0;
                for (int i = 0; i < nPoints; ++i) {
                    record.xyz[k++] = this.raf.readDouble();
                    record.xyz[k++] = this.raf.readDouble();
                    record.xyz[k++] = 0.0;
                }
                break;
            }
            default: {
                throw new IOException("Non-supported Shapefile type " + (Object)((Object)this.shapefileType));
            }
        }
        this.raf.seek(offset0 + 8L + (long)(recLen * 2));
        return record;
    }

    public void rewind() throws IOException {
        this.raf.seek(100L);
    }

    public ShapefileRecord readRecord(int recordNumber, ShapefileRecord pRecord) throws IOException {
        this.readShxFile();
        if (recordNumber < 1 || recordNumber > this.nRecordsInFile) {
            throw new IOException("Record number out of range: " + recordNumber + " for file containing " + this.nRecordsInFile + " records");
        }
        this.raf.seek(this.recordIndex[recordNumber]);
        return this.readNextRecord(pRecord);
    }

    public boolean hasNext() {
        long pos = this.raf.getFilePosition();
        return this.fileLengthInBytes - pos > 8L;
    }

    public int getTotalPointCount() {
        return this.nPointsTotal;
    }

    public int getTotalPartCount() {
        return this.nPartsTotal;
    }

    public ShapefileType getShapefileType() {
        return this.shapefileType;
    }

    public String toString() {
        return "ShapefileReader " + (Object)((Object)this.shapefileType) + " " + this.file.getName();
    }

    public File getCoFile(String extension) {
        int i;
        if (extension == null || extension.length() != 3) {
            return null;
        }
        String path = this.file.getPath();
        int rootLen = path.lastIndexOf(46);
        if (rootLen != path.length() - 4) {
            return null;
        }
        StringBuilder sb = new StringBuilder(++rootLen + 3);
        for (i = 0; i < rootLen; ++i) {
            sb.append(path.charAt(i));
        }
        for (i = 0; i < 3; ++i) {
            char c = path.charAt(i + rootLen);
            char x = extension.charAt(i);
            if (Character.isUpperCase(c) && Character.isLowerCase(x)) {
                sb.append(Character.toUpperCase(x));
                continue;
            }
            if (Character.isLowerCase(c) && Character.isUpperCase(x)) {
                sb.append(Character.toLowerCase(x));
                continue;
            }
            sb.append(x);
        }
        File target = new File(sb.toString());
        if (target.exists()) {
            return target;
        }
        String rootName = path.substring(0, rootLen);
        String test = rootName + extension.toLowerCase();
        target = new File(test);
        if (target.exists()) {
            return target;
        }
        test = rootName + extension.toUpperCase();
        target = new File(test);
        if (target.exists()) {
            return target;
        }
        return null;
    }

    public DbfFileReader getDbfFileReader() throws IOException {
        File target = this.getCoFile("DBF");
        if (target == null) {
            throw new IOException("DBF file not found for " + this.file.getName());
        }
        return new DbfFileReader(target);
    }

    public File getFile() {
        return this.file;
    }

    public long getFileSize() {
        if (this.raf == null) {
            return 0L;
        }
        return this.raf.getFileSize();
    }

    public long getFilePosition() {
        if (this.raf == null) {
            return 0L;
        }
        return this.raf.getFilePosition();
    }
}

