/* --------------------------------------------------------------------
 * Copyright 2016 Gary W. Lucas.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ---------------------------------------------------------------------
 */

 /*
 * -----------------------------------------------------------------------
 *
 * Revision History:
 * Date     Name         Description
 * ------   ---------    -------------------------------------------------
 * 08/2016  G. Lucas     Created
 *
 * Notes:
 *   This class must not keep persistent references to the model or
 *   the composite. Because those objects may represent a very large amount
 *   of memory, we don't want to do anything that will delay them
 *   going out-of-scope. This consideration is especially important because
 *   the UI may keep instances of this class around in "stand by" so that
 *   it can be presented more quickly.
 *
 * -----------------------------------------------------------------------
 */
package org.tinfour.demo.viewer;

import java.awt.Color;
import java.awt.Component;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.text.ParseException;
import javax.swing.JDialog;
import javax.swing.JTextField;
import org.tinfour.common.Vertex;
import org.tinfour.demo.utils.TextCoordCartesian;
import org.tinfour.demo.utils.TextCoordGeo;
import org.tinfour.demo.viewer.backplane.IModel;
import org.tinfour.demo.viewer.backplane.IModelChangeListener;
import org.tinfour.demo.viewer.backplane.MvComposite;
import org.tinfour.utils.LinearUnits;

/**
 * Provides UI for showing viewing options
 */
@SuppressWarnings("PMD")  // due to non-compliant, automatically generated code
class ZoomToFeaturePanel extends javax.swing.JPanel implements IModelChangeListener {

  private static final long serialVersionUID = 1L;
  private DataViewingPanel dvPanel;

  // variables for tracking current vertex
  private Vertex currentFeature;
  private double currentWidth;
  private Point2D currentCoordinates;

  private final TextCoordGeo latLonParser = new TextCoordGeo();
  private final TextCoordCartesian cartParser = new TextCoordCartesian();

  /**
   * Creates new form ViewOptions
   */
  ZoomToFeaturePanel() {
    initComponents();

  }

  /**
   * Sets a reference to the current data-viewing panel.
   *
   * @param dvPanel a valid instance.
   */
  void setDataViewingPanel(DataViewingPanel dvPanel) {
    this.dvPanel = dvPanel;
    MvComposite mvComposite = dvPanel.getMvComposite();
    IModel model = mvComposite.getModel();
    AffineTransform p2m = dvPanel.getPanelToModelTransform();
    if (p2m == null) {
      return;
    }
    int xP = dvPanel.getWidth() / 2;
    int yP = dvPanel.getHeight() / 2;
    Point2D pPoint = new Point2D.Double(xP, yP);
    Point2D mPoint = new Point2D.Double();
    p2m.transform(pPoint, mPoint);
  }

  void registerDataViewingPanel(DataViewingPanel dvPanel) {
    this.dvPanel = dvPanel;
    dvPanel.addModelChangeListener(this);
    IModel model = dvPanel.getModel();
    if (model != null && model.isLoaded()) {
      this.transferValuesFromModel(model);
    }
    setControlsForInputState();
  }

  private IModel getModel() {
    MvComposite c = dvPanel.getMvComposite();
    if (c != null) {
      return c.getModel();
    }
    return null;
  }

  private void setDouble(JTextField field, String fmt, double v) {
    // if the value is NaN, just leave field alone
    if (!Double.isNaN(v)) {
      if (Math.floor(v + 1.0e-6) == (int) v) {
        field.setText(Integer.toString((int) v));
      } else {
        field.setText(String.format(fmt, v));
      }
    }
  }

  private double extractField(JTextField field, double value) {
    double d = extractDouble(field);
    if (Double.isNaN(d)) {
      return value;
    }
    return d;
  }

  private double extractDouble(JTextField field) {
    try {
      String s = field.getText();
      if (s != null) {
        s = s.trim();
        if (!s.isEmpty()) {
          return Double.parseDouble(s.trim());
        }
      }
    } catch (NumberFormatException nex) {
    }
    return Double.NaN;
  }

  private void hideDialog() {
    Component c = this;
    do {
      c = c.getParent();
    } while (!(c instanceof JDialog));
    JDialog jd = (JDialog) c;
    jd.setVisible(false);

  }

  /**
   * This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  @SuppressWarnings("unchecked")
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {

    optionsPanel = new javax.swing.JPanel();
    coordinatesLabel = new javax.swing.JLabel();
    featureLabel = new javax.swing.JLabel();
    widthLabel = new javax.swing.JLabel();
    coordinatesTextField = new javax.swing.JTextField();
    featureTextField = new javax.swing.JTextField();
    widthTextField = new javax.swing.JTextField();
    unitsLabel = new javax.swing.JLabel();
    actionsPanel = new javax.swing.JPanel();
    applyButton = new javax.swing.JButton();
    cancelButton = new javax.swing.JButton();
    okayButton = new javax.swing.JButton();
    resetButton = new javax.swing.JButton();

    optionsPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());

    coordinatesLabel.setText("Coordinates:");

    featureLabel.setText("Feature:");

    widthLabel.setText("Width:");

    coordinatesTextField.setColumns(24);
    coordinatesTextField.setToolTipText("Enter a pair of coordinates separated by a space");
    coordinatesTextField.addKeyListener(new java.awt.event.KeyAdapter() {
      public void keyReleased(java.awt.event.KeyEvent evt) {
        coordinatesTextFieldKeyReleased(evt);
      }
    });

    featureTextField.setColumns(24);
    featureTextField.setText(" ");
    featureTextField.setToolTipText("Enter a feature ID to center display");
    featureTextField.addFocusListener(new java.awt.event.FocusAdapter() {
      public void focusLost(java.awt.event.FocusEvent evt) {
        featureTextFieldFocusLost(evt);
      }
    });
    featureTextField.addKeyListener(new java.awt.event.KeyAdapter() {
      public void keyReleased(java.awt.event.KeyEvent evt) {
        featureTextFieldKeyReleased(evt);
      }
    });

    widthTextField.setColumns(24);
    widthTextField.setText(" ");
    widthTextField.setToolTipText("Enter width of display area");
    widthTextField.addKeyListener(new java.awt.event.KeyAdapter() {
      public void keyReleased(java.awt.event.KeyEvent evt) {
        widthTextFieldKeyReleased(evt);
      }
    });

    unitsLabel.setText("Unspecified Units");

    javax.swing.GroupLayout optionsPanelLayout = new javax.swing.GroupLayout(optionsPanel);
    optionsPanel.setLayout(optionsPanelLayout);
    optionsPanelLayout.setHorizontalGroup(
      optionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(optionsPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addGroup(optionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
          .addComponent(coordinatesLabel)
          .addComponent(featureLabel)
          .addComponent(widthLabel))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addGroup(optionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
          .addComponent(coordinatesTextField)
          .addComponent(featureTextField)
          .addGroup(optionsPanelLayout.createSequentialGroup()
            .addComponent(widthTextField, javax.swing.GroupLayout.PREFERRED_SIZE, 97, javax.swing.GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
            .addComponent(unitsLabel)))
        .addContainerGap(84, Short.MAX_VALUE))
    );
    optionsPanelLayout.setVerticalGroup(
      optionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(optionsPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addGroup(optionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(featureLabel)
          .addComponent(featureTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addGroup(optionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(coordinatesLabel)
          .addComponent(coordinatesTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addGroup(optionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(widthLabel)
          .addComponent(widthTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(unitsLabel))
        .addContainerGap(26, Short.MAX_VALUE))
    );

    applyButton.setText("Apply");
    applyButton.setToolTipText("Apply settings");
    applyButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        applyButtonActionPerformed(evt);
      }
    });

    cancelButton.setText("Cancel");
    cancelButton.setToolTipText("Close dialog without applying settings");
    cancelButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        cancelButtonActionPerformed(evt);
      }
    });

    okayButton.setText("OK");
    okayButton.setToolTipText("Apply settings and close dialog");
    okayButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        okButtonActionPerformed(evt);
      }
    });

    resetButton.setText("Reset");
    resetButton.setToolTipText("Reset coordinates using current model and view");
    resetButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        resetButtonActionPerformed(evt);
      }
    });

    javax.swing.GroupLayout actionsPanelLayout = new javax.swing.GroupLayout(actionsPanel);
    actionsPanel.setLayout(actionsPanelLayout);
    actionsPanelLayout.setHorizontalGroup(
      actionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(actionsPanelLayout.createSequentialGroup()
        .addComponent(resetButton)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        .addComponent(okayButton)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(cancelButton)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(applyButton)
        .addContainerGap())
    );

    actionsPanelLayout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {applyButton, cancelButton, okayButton, resetButton});

    actionsPanelLayout.setVerticalGroup(
      actionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, actionsPanelLayout.createSequentialGroup()
        .addGap(0, 0, Short.MAX_VALUE)
        .addGroup(actionsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(okayButton)
          .addComponent(cancelButton)
          .addComponent(applyButton)
          .addComponent(resetButton)))
    );

    actionsPanelLayout.linkSize(javax.swing.SwingConstants.VERTICAL, new java.awt.Component[] {applyButton, cancelButton, okayButton, resetButton});

    javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
    this.setLayout(layout);
    layout.setHorizontalGroup(
      layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(layout.createSequentialGroup()
        .addContainerGap()
        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
          .addComponent(actionsPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
          .addComponent(optionsPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        .addContainerGap())
    );
    layout.setVerticalGroup(
      layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(layout.createSequentialGroup()
        .addContainerGap()
        .addComponent(optionsPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 23, Short.MAX_VALUE)
        .addComponent(actionsPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addContainerGap())
    );
  }// </editor-fold>//GEN-END:initComponents

  private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_okButtonActionPerformed
    this.transferValuesToDataViewPanel();
    hideDialog();

  }//GEN-LAST:event_okButtonActionPerformed

  private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
    hideDialog();
  }//GEN-LAST:event_cancelButtonActionPerformed

  private void applyButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_applyButtonActionPerformed
    this.transferValuesToDataViewPanel();
  }//GEN-LAST:event_applyButtonActionPerformed

  private void transferValuesToDataViewPanel() {
    IModel model = getModel();
    if (model != null && this.currentCoordinates != null && currentWidth > 0) {
      double x = currentCoordinates.getX();
      double y = currentCoordinates.getY();
      dvPanel.zoomToModelPosition(x, y, currentWidth);
    }
  }

  private void coordinatesTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_coordinatesTextFieldKeyReleased
    IModel model = getModel();
    if (model == null) {
      return;
    }

    currentFeature = null;
    featureTextField.setText("");
    featureTextField.setForeground(Color.black);
    String s = coordinatesTextField.getText();
    if (s == null || s.isEmpty()) {
      coordinatesTextField.setForeground(Color.black);
    } else {
      try {
        double[] c = new double[2];
        if (model.isCoordinateSystemGeographic()) {
          double[] geo = latLonParser.parse(s);
          model.geo2xy(geo[0], geo[1], c);
        } else {
          c = cartParser.parse(s);
        }
        currentCoordinates = new Point2D.Double(c[0], c[1]);
        coordinatesTextField.setForeground(Color.black);
      } catch (IllegalArgumentException | ParseException ex) {
        currentCoordinates = null;
        coordinatesTextField.setForeground(Color.red);
      }
    }
    setControlsForInputState();
  }//GEN-LAST:event_coordinatesTextFieldKeyReleased

  private void featureTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_featureTextFieldKeyReleased
    IModel model = getModel();
    if (model == null) {
      return;
    }

    currentFeature = null;
    String s = featureTextField.getText();
    if (s == null || s.isEmpty()) {
      featureTextField.setForeground(Color.black);
    } else {
      Integer index = -1;
      try {
        index = Integer.parseInt(s.trim());
        currentFeature = model.getVertexForIndex(index);
        if (currentFeature == null) {
          featureTextField.setForeground(Color.red);
        } else {
          featureTextField.setForeground(Color.black);
          transferCurrentFeatureToPositionField(model);
          //  if (evt.getKeyCode() == java.awt.event.KeyEvent.VK_ENTER) {
          //    transferCurrentFeatureToPositionField(model);
          //  }
        }
      } catch (NumberFormatException nex) {
        featureTextField.setForeground(Color.red);
      }
    }
    setControlsForInputState();
  }//GEN-LAST:event_featureTextFieldKeyReleased

  private void featureTextFieldFocusLost(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_featureTextFieldFocusLost
    IModel model = getModel();
    if (model == null) {
      return;
    }
//    if (currentFeature != null) {
//      transferCurrentFeatureToPositionField(model);
//    }
  }//GEN-LAST:event_featureTextFieldFocusLost

  private void widthTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_widthTextFieldKeyReleased
    IModel model = getModel();
    String s = widthTextField.getText();
    currentWidth = 0;
    if (model != null && s != null && !s.isEmpty()) {
      try {
        double d = Double.parseDouble(s);
        if (d > 0) {
          currentWidth = d;
          widthTextField.setForeground(Color.black);
        }
      } catch (NumberFormatException nex) {
        widthTextField.setForeground(Color.red);
      }
      setControlsForInputState();
    }
  }//GEN-LAST:event_widthTextFieldKeyReleased

  private void resetButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_resetButtonActionPerformed
    IModel model = getModel();
    if(model!=null){
    transferValuesFromModel(model);
    }
  }//GEN-LAST:event_resetButtonActionPerformed

  /**
   * Transfers position data from current feature, only invoked when
   * model is non-null.
   *
   * @param model a valid model.
   */
  private void transferCurrentFeatureToPositionField(IModel model) {
    double vx = currentFeature.getX();
    double vy = currentFeature.getY();
    String s;
    if (model.isCoordinateSystemGeographic()) {
      double[] geo = new double[2];
      model.xy2geo(vx, vy, geo);
      s = latLonParser.format(geo[0], geo[1]);
    } else {
      s = cartParser.format(vx, vy);
    }

    coordinatesTextField.setText(s);
    coordinatesTextField.setForeground(Color.black);
    currentCoordinates = new Point2D.Double(vx, vy);

    double modelWidth = model.getMaxX() - model.getMinX();
    double nps = model.getNominalPointSpacing();
    currentWidth = nps * 6;
    if (currentWidth > modelWidth) {
      currentWidth = modelWidth;
    }
    s = String.format("%f", currentWidth);
    this.widthTextField.setText(s);
  }

  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JPanel actionsPanel;
  private javax.swing.JButton applyButton;
  private javax.swing.JButton cancelButton;
  private javax.swing.JLabel coordinatesLabel;
  private javax.swing.JTextField coordinatesTextField;
  private javax.swing.JLabel featureLabel;
  private javax.swing.JTextField featureTextField;
  private javax.swing.JButton okayButton;
  private javax.swing.JPanel optionsPanel;
  private javax.swing.JButton resetButton;
  private javax.swing.JLabel unitsLabel;
  private javax.swing.JLabel widthLabel;
  private javax.swing.JTextField widthTextField;
  // End of variables declaration//GEN-END:variables

  @Override
  public void modelRemoved() {
    clearInputDataElements();
    featureTextField.setEnabled(false);
    coordinatesTextField.setEnabled(false);
    widthTextField.setEnabled(false);
    okayButton.setEnabled(false);
    applyButton.setEnabled(false);
    resetButton.setEnabled(false);
    this.repaint();
  }

  @Override
  public void modelAdded(IModel model) {
    this.transferValuesFromModel(model);
    setControlsForInputState();
    resetButton.setEnabled(true);
  }

  /**
   * Clears values for both input-related member elements and UI components.
   */
  private void clearInputDataElements() {
    currentFeature = null;
    currentWidth = 0;
    currentCoordinates = null;
    featureTextField.setText("");
    featureTextField.setForeground(Color.black);
    coordinatesTextField.setText("");
    coordinatesTextField.setForeground(Color.black);
    widthTextField.setText("");
    widthTextField.setForeground(Color.black);
  }

  void transferValuesFromPanel() {
    IModel model = dvPanel.getModel();
    transferValuesFromModel(model);
  }

  /**
   * Determines if the current model-view composite associated with the
   * DataViewPanel is valid and, if so, transfers its data to the UI.
   * The return value indicates whether the validity test was successful.
   * The application should not raise the dialog if the composite is
   * not valid.
   *
   * @param dvPanel a valid instance
   * @return true if the panel may be raised; otherwise, false.
   */
  void transferValuesFromModel(IModel model) {
    if (model == null || !model.isLoaded()) {
      clearInputDataElements();
      return;
    }

    AffineTransform p2m = dvPanel.getPanelToModelTransform();
    if (p2m == null) {
      return;
    }

    currentFeature = null;
    featureTextField.setForeground(Color.black);
    featureTextField.setText("");
    featureTextField.setEnabled(true);

    String s;
    int xP = dvPanel.getWidth() / 2;
    int yP = dvPanel.getHeight() / 2;
    Point2D pPoint = new Point2D.Double(xP, yP);
    Point2D mPoint = new Point2D.Double();
    p2m.transform(pPoint, mPoint);
    currentCoordinates = mPoint;

    if (model.isCoordinateSystemGeographic()) {
      double[] g = new double[2];
      model.xy2geo(mPoint.getX(), mPoint.getY(), g);
      s = latLonParser.format(g[0], g[1]);
      coordinatesTextField.setToolTipText("Enter geographic coordinate pair separated by slash or space");
    } else {
      s = cartParser.format(mPoint.getX(), mPoint.getY());
      coordinatesTextField.setToolTipText("Enter Cartesian coordinate pair separated by space");
    }

    coordinatesTextField.setText(s);
    coordinatesTextField.setEnabled(true);

    double scale = Math.sqrt(Math.abs(p2m.getDeterminant()));
    double pWidth = dvPanel.getWidth();
    currentWidth = pWidth * scale;
    s = String.format("%f", currentWidth);
    widthTextField.setText(s);
    widthTextField.setEnabled(true);

    LinearUnits linearUnits =model.getLinearUnits();
    if(linearUnits == LinearUnits.UNKNOWN){
      unitsLabel.setText("Unspecified units");
    }else{
      unitsLabel.setText(linearUnits.getName());
    }

    setControlsForInputState();
  }

  private void setControlsForInputState() {
    if (currentCoordinates != null && currentWidth > 0) {
      okayButton.setEnabled(true);
      applyButton.setEnabled(true);
    } else {
      okayButton.setEnabled(false);
      applyButton.setEnabled(false);
    }
  }

}
