/*
 * Decompiled with CFR 0.152.
 */
package org.tinfour.demo.examples.lake;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import org.tinfour.common.GeometricOperations;
import org.tinfour.common.IConstraint;
import org.tinfour.common.IIncrementalTin;
import org.tinfour.common.IQuadEdge;
import org.tinfour.common.PolygonConstraint;
import org.tinfour.common.SimpleTriangle;
import org.tinfour.common.Thresholds;
import org.tinfour.common.Vertex;
import org.tinfour.demo.examples.lake.BathymetryData;
import org.tinfour.semivirtual.SemiVirtualIncrementalTin;
import org.tinfour.standard.IncrementalTin;
import org.tinfour.utils.KahanSummation;
import org.tinfour.utils.TriangleCollector;

public class LakeVolumeExample {
    public static void main(String[] args) {
        String dbfFieldForDepth;
        File inputIslandFile;
        File inputLakeFile;
        File inputSoundingsFile;
        boolean useSilsbeSample = false;
        String path = ".";
        for (int i = 0; i < args.length; ++i) {
            if ("-Silsbe".equalsIgnoreCase(args[i])) {
                useSilsbeSample = true;
            }
            if (!"-input".equalsIgnoreCase(args[i])) continue;
            if (i == args.length - 1) {
                throw new IllegalArgumentException("-input option requires a folder path");
            }
            path = args[i + 1];
        }
        if (useSilsbeSample) {
            File folder = path == null ? new File("LakeVictoriaSilsbe") : new File(path);
            inputSoundingsFile = new File(folder, "finbath.shp");
            inputLakeFile = new File(folder, "LAKE.SHP");
            inputIslandFile = new File(folder, "ISLANDS.SHP");
            dbfFieldForDepth = "DEPTH";
        } else {
            File folder = path == null ? new File("LakeVictoriaSalisburyUniversity") : new File(path);
            inputSoundingsFile = new File(folder, "LV_Bathymetry_Points_V7.shp");
            inputLakeFile = new File(folder, "LakeVictoriaShoreline.shp");
            inputIslandFile = null;
            dbfFieldForDepth = "Z";
        }
        try {
            BathymetryData bathyData = new BathymetryData(inputSoundingsFile, inputLakeFile, inputIslandFile, dbfFieldForDepth);
            bathyData.printSummary(System.out);
            LakeVolumeExample lvCalc = new LakeVolumeExample();
            lvCalc.processVolume(System.out, bathyData);
        }
        catch (IOException ioex) {
            System.out.println("Exception processing lake volume files, " + ioex.getMessage());
            ioex.printStackTrace(System.out);
        }
    }

    public void processVolume(PrintStream ps, BathymetryData data) {
        List<Vertex> soundings = data.getSoundings();
        List<PolygonConstraint> lakeConstraints = data.getLakeConstraints();
        List<PolygonConstraint> islandConstraints = data.getIslandConstraints();
        for (PolygonConstraint con : lakeConstraints) {
            con.setApplicationData((Object)true);
        }
        for (PolygonConstraint con : islandConstraints) {
            con.setApplicationData((Object)false);
        }
        ArrayList<PolygonConstraint> allConstraints = new ArrayList<PolygonConstraint>();
        allConstraints.addAll(lakeConstraints);
        allConstraints.addAll(islandConstraints);
        long time0 = System.nanoTime();
        Object tin = soundings.size() < 500000 ? new IncrementalTin(1.0) : new SemiVirtualIncrementalTin(1.0);
        tin.add(soundings, null);
        tin.addConstraints(allConstraints, true);
        long time1 = System.nanoTime();
        LakeData results = new LakeData((IIncrementalTin)tin);
        TriangleCollector.visitSimpleTriangles((IIncrementalTin)tin, (Consumer)results);
        long time2 = System.nanoTime();
        double lakeArea = this.getAreaSum(lakeConstraints);
        double islandArea = this.getAreaSum(islandConstraints);
        double lakePerimeter = this.getPerimeterSum(lakeConstraints);
        double islandPerimeter = this.getPerimeterSum(islandConstraints);
        double netArea = lakeArea - islandArea;
        double totalShore = lakePerimeter + islandPerimeter;
        ps.format("%nData from Shapefiles%n", new Object[0]);
        ps.format("  Lake area        %10.8e %,20.0f m2 %9.1f km2%n", lakeArea, lakeArea, lakeArea / 1000000.0);
        ps.format("  Island area      %10.8e %,20.0f m2 %9.1f km2%n", islandArea, islandArea, islandArea / 1000000.0);
        ps.format("  Net area (water) %10.8e %,20.0f m2 %9.1f km2%n", netArea, netArea, netArea / 1000000.0);
        ps.format("  Lake shoreline   %10.8e %,20.0f m  %9.1f km%n", lakePerimeter, lakePerimeter, lakePerimeter / 1000.0);
        ps.format("  Island shoreline %10.8e %,20.0f m  %9.1f km%n", islandPerimeter, islandPerimeter, islandPerimeter / 1000.0);
        ps.format("  Total shoreline  %10.8e %,20.0f m  %9.1f km%n", totalShore, totalShore, totalShore / 1000.0);
        ps.format("  N Islands        %d%n", islandConstraints.size());
        double volume = results.getVolume();
        double surfArea = results.getSurfaceArea();
        double avgDepth = volume / surfArea;
        double sampleSpacing = this.estimateSampleSpacing((IIncrementalTin)tin, results);
        ps.format("%nComputations from Constrained Delaunay Triangulation%n", new Object[0]);
        ps.format("  Volume           %10.8e %,20.0f m3 %9.1f km3%n", volume, volume, volume / 1.0E9);
        ps.format("  Surface Area     %10.8e %,20.0f m2 %9.1f km2%n", surfArea, surfArea, surfArea / 1000000.0);
        ps.format("  Avg depth       %5.2f m%n", avgDepth);
        ps.format("  N Triangles     %d%n", results.nTriangles);
        ps.format("  Sample Spacing %8.2f m%n", sampleSpacing);
        ps.format("%n%n%n", new Object[0]);
        ps.format("Time to load data           %7.1f ms%n", (double)data.getTimeToLoadData() / 1000000.0);
        ps.format("Time to build TIN           %7.1f ms%n", (double)(time1 - time0) / 1000000.0);
        ps.format("Time to compute lake volume %7.1f ms%n", (double)(time2 - time1) / 1000000.0);
        ps.format("Time for all analysis       %7.1f ms%n", (double)(time2 - time0) / 1000000.0);
        ps.format("Time for all operations     %7.1f ms%n", (double)(data.getTimeToLoadData() + time2 - time0) / 1000000.0);
    }

    private double getAreaSum(List<PolygonConstraint> constraints) {
        KahanSummation areaSum = new KahanSummation();
        for (PolygonConstraint con : constraints) {
            areaSum.add(con.getArea());
        }
        return areaSum.getSum();
    }

    private double getPerimeterSum(List<PolygonConstraint> constraints) {
        KahanSummation perimeterSum = new KahanSummation();
        for (PolygonConstraint con : constraints) {
            perimeterSum.add(con.getLength());
        }
        return perimeterSum.getSum();
    }

    private double estimateSampleSpacing(IIncrementalTin tin, LakeData lakeData) {
        KahanSummation sumLength = new KahanSummation();
        int n = 0;
        for (IQuadEdge e : tin.edges()) {
            if (!lakeData.isWater(e)) continue;
            Vertex a = e.getA();
            Vertex b = e.getB();
            if (a.isConstraintMember() || b.isConstraintMember()) continue;
            ++n;
            sumLength.add(e.getLength());
        }
        return sumLength.getSum() / (double)n;
    }

    private static class LakeData
    implements Consumer<SimpleTriangle> {
        boolean[] water;
        final GeometricOperations geoOp;
        int nTriangles;
        KahanSummation volumeSum = new KahanSummation();
        KahanSummation areaSum = new KahanSummation();

        LakeData(IIncrementalTin tin) {
            List constraintsFromTin = tin.getConstraints();
            this.water = new boolean[constraintsFromTin.size()];
            for (IConstraint con : constraintsFromTin) {
                this.water[con.getConstraintIndex()] = (Boolean)con.getApplicationData();
            }
            Thresholds thresholds = tin.getThresholds();
            this.geoOp = new GeometricOperations(thresholds);
        }

        @Override
        public void accept(SimpleTriangle t) {
            Boolean appData;
            IConstraint constraint = t.getContainingRegion();
            if (constraint instanceof PolygonConstraint && (appData = (Boolean)constraint.getApplicationData()).booleanValue()) {
                IQuadEdge a = t.getEdgeA();
                IQuadEdge b = t.getEdgeB();
                IQuadEdge c = t.getEdgeC();
                Vertex vA = a.getA();
                Vertex vB = b.getA();
                Vertex vC = c.getA();
                double zA = vA.getZ();
                double zB = vB.getZ();
                double zC = vC.getZ();
                double zMean = (zA + zB + zC) / 3.0;
                double area = this.geoOp.area(vA, vB, vC);
                ++this.nTriangles;
                this.volumeSum.add(zMean * area);
                this.areaSum.add(area);
            }
        }

        boolean isWater(IQuadEdge edge) {
            if (edge.isConstrainedRegionBorder()) {
                return false;
            }
            if (edge.isConstrainedRegionInterior()) {
                int index = edge.getConstraintIndex();
                return this.water[index];
            }
            return false;
        }

        double getVolume() {
            return Math.abs(this.volumeSum.getSum());
        }

        double getSurfaceArea() {
            return this.areaSum.getSum();
        }
    }
}

