/**
 *  Copyright (c) 1997-2013, tinygroup.org (luo_guo@live.cn).
 *
 *  Licensed under the GPL, Version 3.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.gnu.org/licenses/gpl.html
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * --------------------------------------------------------------------------
 *  版权 (c) 1997-2013, tinygroup.org (luo_guo@live.cn).
 *
 *  本开源软件遵循 GPL 3.0 协议;
 *  如果您不遵循此协议，则不被允许使用此文件。
 *  你可以从下面的地址获取完整的协议文本
 *
 *       http://www.gnu.org/licenses/gpl.html
 */
package org.tinygroup.threadpool.impl;

import org.tinygroup.logger.Logger;
import org.tinygroup.logger.LoggerFactory;
import org.tinygroup.pool.impl.PoolDoubleList;
import org.tinygroup.queue.PriorityQueue;
import org.tinygroup.queue.impl.PriorityQueueImpl;
import org.tinygroup.threadpool.ThreadPool;
import org.tinygroup.threadpool.TaskThread;
import org.tinygroup.threadpool.ThreadProvider;

public class ThreadPoolImpl implements ThreadPool {
	private static Logger logger = LoggerFactory.getLogger(ThreadPoolImpl.class);
	//主线程
	private MainThread mainThread;
	//线程池
	private PoolDoubleList<TaskThread> threadPool = new PoolDoubleList<TaskThread>();
	private int initSize = DEFAULT_INIT_SIZE;
	private int maxSize = DEFAULT_MAX_SIZE;
	private int increaseStep = DEFAULT_INCREASE_STEP;
	private boolean testBeforeBorrow;
	//任务队列
	private PriorityQueue<Runnable> priorityQueue = new PriorityQueueImpl<Runnable>();
	private ThreadGroup group = null;

	public PoolDoubleList<TaskThread> getTaskThreadPool() {
		return threadPool;
	}

	public int getInitSize() {
		return initSize;
	}

	public void setInitSize(int initSize) {
		this.initSize = initSize;
	}

	public int getMaxSize() {
		return maxSize;
	}

	public void setMaxSize(int maxSize) {
		this.maxSize = maxSize;
	}

	public int getIncreaseStep() {
		return increaseStep;
	}

	public void setIncreaseStep(int increaseStep) {
		this.increaseStep = increaseStep;
	}

	public PriorityQueue<Runnable> getPriorityQueue() {
		return priorityQueue;
	}

	public ThreadPoolImpl(String name) {
		group = new ThreadGroup(name);
		mainThread = new MainThread(group);
		mainThread.setName("" + mainThread.getId());
		priorityQueue.setPriorityIncreaseStrategy(null);
	}

	public void stopThreadPool() {
		mainThread.setStopThreadPool(true);
	}

	public void startThreadPool() {
		threadPool.initPool(initSize, maxSize, increaseStep, testBeforeBorrow,
				new ThreadProvider(this, group));
		threadPool.start();
		mainThread.start();
	}

	public void interrupt() {

	}

	/**
	 * 添加任务
	 */
	public boolean addTask(Runnable task) {
		if (mainThread.isStopThreadPool()) {
			return false;
		} else {
			return priorityQueue.offer(task);
		}
	}

	public class MainThread extends Thread {
		private boolean stopThreadPool = false;

		public MainThread(ThreadGroup group) {
			super(group, (Runnable) null);
		}

		public boolean isStopThreadPool() {
			return stopThreadPool;
		}

		public void setStopThreadPool(boolean stopThreadPool) {
			this.stopThreadPool = stopThreadPool;
		}

		public void run() {
			while (!stopThreadPool) {

				//如果有任务，则执行任务
				while (threadPool.getUsingSize() < getMaxSize()
						&& !priorityQueue.isEmpty()) {
					try {
						TaskThread t = threadPool.borrowObject();
						Runnable r = priorityQueue.poll();
						synchronized (t) {
							t.executeTask(r);
							t.notify();
						}
					} catch (Exception e) {
						logger.errorMessage(e.getMessage(), e);
					}

				}
				try {
					sleep(DEFAULT_SLEEP_INTERVAL);
				} catch (InterruptedException e) {
					stopThreadPool = true;
				}
			}
			// STOP 子线程
			threadPool.stop();

		}
	}

	public int size() {
		return priorityQueue.getSize();
	}
}
