/**
 *  Copyright (c) 1997-2013, tinygroup.org (luo_guo@live.cn).
 *
 *  Licensed under the GPL, Version 3.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.gnu.org/licenses/gpl.html
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * --------------------------------------------------------------------------
 *  版权 (c) 1997-2013, tinygroup.org (luo_guo@live.cn).
 *
 *  本开源软件遵循 GPL 3.0 协议;
 *  如果您不遵循此协议，则不被允许使用此文件。
 *  你可以从下面的地址获取完整的协议文本
 *
 *       http://www.gnu.org/licenses/gpl.html
 */
package org.tinygroup.weblayer;

import static org.tinygroup.logger.LogLevel.INFO;

import java.io.InputStream;
import java.net.MalformedURLException;
import java.util.Enumeration;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import org.tinygroup.appconfig.AppConfigManager;
import org.tinygroup.application.Application;
import org.tinygroup.fileresolver.FileResolver;
import org.tinygroup.fileresolver.impl.FileResolverImpl;
import org.tinygroup.fileresolver.impl.SpringBeansFileProcessor;
import org.tinygroup.logger.LogLevel;
import org.tinygroup.logger.Logger;
import org.tinygroup.logger.LoggerFactory;
import org.tinygroup.parser.filter.PathFilter;
import org.tinygroup.springutil.SpringUtil;
import org.tinygroup.weblayer.listener.ServletContextHolder;
import org.tinygroup.xmlparser.node.XmlNode;
import org.tinygroup.xmlparser.parser.XmlStringParser;

public class ApplicationStartupListener implements ServletContextListener {
	private static Logger logger = LoggerFactory
			.getLogger(ApplicationStartupListener.class);
	private Application application;
	static String configString = null;
	static {
		configString = loadConfigContent();
	}

	public void contextDestroyed(ServletContextEvent servletContextEvent) {
		logger.logMessage(LogLevel.INFO, "WEB 应用停止中...");
		application.stop();
		SpringUtil.destory();//关闭spring容器
		logger.logMessage(LogLevel.INFO, "WEB 应用停止完成。");
	}

	@SuppressWarnings("unchecked")
	public void contextInitialized(ServletContextEvent servletContextEvent) {
		ServletContextHolder.setServletContext(servletContextEvent
				.getServletContext());
		Enumeration<String> enumeration = servletContextEvent
				.getServletContext().getAttributeNames();
		logger.logMessage(LogLevel.INFO, "WEB环境属性开始");
		while (enumeration.hasMoreElements()) {
			String key = enumeration.nextElement();
			logger.logMessage(LogLevel.INFO, "{0}=[{1}]", key,
					servletContextEvent.getServletContext().getAttribute(key));
		}
		logger.logMessage(LogLevel.INFO, "WEB 应用启动中...");
		loadSpringBeans();

		application = SpringUtil.getBean("application");
		logger.logMessage(LogLevel.INFO, "WEB 应用信息：[{0}]", servletContextEvent
				.getServletContext().getServerInfo());
		String webRootPath = servletContextEvent.getServletContext()
				.getRealPath("/");
		if (webRootPath == null) {
			try {
				webRootPath = servletContextEvent.getServletContext()
						.getResource("/").getFile();
			} catch (MalformedURLException e) {
				logger.errorMessage("获取WEBROOT失败！", e);
			}
		}
		logger.logMessage(LogLevel.INFO, "TINY_WEBROOT：[{0}]", webRootPath);
		application.getContext().put("TINY_WEBROOT", webRootPath);
		logger.logMessage(LogLevel.INFO, "应用参数<TINY_WEBROOT>=<{}>", webRootPath);

		logger.logMessage(LogLevel.INFO, "ServerContextName：[{0}]",
				servletContextEvent.getServletContext().getServletContextName());
		logger.logMessage(LogLevel.INFO, "WEB环境属性结束");
		AppConfigManager appConfigManager = SpringUtil
				.getBean("appConfigManager");

		appConfigManager.loadConfig(loadConfigContent());
		appConfigManager.distributeConfig();
		application.start();

		logger.logMessage(LogLevel.INFO, "WEB 应用启动完成。");
	}

	private void loadSpringBeans() {
		logger.logMessage(LogLevel.INFO, "加载Spring Bean文件开始...");
		FileResolver fileResolver = new FileResolverImpl();
		loadFileResolverConfig(fileResolver);
		fileResolver.addFileProcessor(new SpringBeansFileProcessor());
		fileResolver.resolve();
		logger.logMessage(LogLevel.INFO, "加载Spring Bean文件结束。");
	}

	private void loadFileResolverConfig(FileResolver fileResolver) {
		XmlStringParser parser = new XmlStringParser();
		XmlNode root = parser.parse(loadConfigContent()).getRoot();
		PathFilter<XmlNode> filter = new PathFilter<XmlNode>(root);
		String resolve = filter.findNode(
				"/application/file-resolver-configuration").getAttribute(
				"resolve-classpath");
		if (resolve == null || resolve.length() == 0) {
			resolve = "true";
		}

		fileResolver.setClassPathResolve(Boolean.parseBoolean(resolve));
		List<XmlNode> skipList = filter
				.findNodeList("/application/file-resolver-configuration/skip-patterns/skip-pattern");
		for (XmlNode node : skipList) {
			fileResolver.addSkipPathPattern(node.getAttribute("pattern"));
		}
		List<XmlNode> classPathList = filter
				.findNodeList("/application/file-resolver-configuration/class-paths/class-path");
		for (XmlNode classPath : classPathList) {
			fileResolver.addManualClassPath(classPath
					.getAttribute("path"));
		}
		for (XmlNode node : skipList) {
			fileResolver.addSkipPathPattern(node.getAttribute("pattern"));
		}
	}

	private static String loadConfigContent() {
		try {
			logger.logMessage(INFO, "配置文件<{0}>加载开始...",
					AppConfigManager.APPLICATION_CONFIGURATION_FILE);
			InputStream inputStream = ApplicationStartupListener.class
					.getResourceAsStream(AppConfigManager.APPLICATION_CONFIGURATION_FILE);
			byte[] bytes = new byte[inputStream.available()];
			inputStream.read(bytes);
			inputStream.close();
			String content = new String(bytes, "UTF-8");
			logger.logMessage(INFO, "配置文件<{0}>加载完成。",
					AppConfigManager.APPLICATION_CONFIGURATION_FILE);
			XmlNode configurationXmlNode = new XmlStringParser().parse(content)
					.getRoot();
			PathFilter<XmlNode> filter = new PathFilter<XmlNode>(
					configurationXmlNode);
			List<XmlNode> propertyList = filter
					.findNodeList("/application/application-properties/property");
			for (XmlNode property : propertyList) {
				String name = property.getAttribute("name");
				String value = property.getAttribute("value");
				content = replace(content, name, value);
			}

			return content;
		} catch (Exception e) {
			logger.errorMessage("配置文件<{0}>加载时发生错误。", e,
					AppConfigManager.APPLICATION_CONFIGURATION_FILE);
			throw new RuntimeException(e);
		}
	}

	private static String replace(String content, String name, String value) {
		Pattern pattern = Pattern.compile("[{]" + name + "[}]");
		Matcher matcher = pattern.matcher(content);
		StringBuffer buf = new StringBuffer();
		int curpos = 0;
		while (matcher.find()) {
			buf.append(content.substring(curpos, matcher.start()));
			curpos = matcher.end();
			buf.append(value);
			continue;
		}
		buf.append(content.substring(curpos));
		return buf.toString();
	}
}
