package org.tkit.rhpam.quarkus.processlog.domain.daos;

import lombok.extern.slf4j.Slf4j;
import org.hibernate.service.spi.ServiceException;
import org.tkit.quarkus.jpa.daos.AbstractDAO;
import org.tkit.rhpam.quarkus.processlog.domain.models.FailedStep;
import org.tkit.rhpam.quarkus.processlog.domain.models.FailedStepSearchCriteria;
import org.tkit.rhpam.quarkus.processlog.domain.models.FFailedStep_;

import javax.enterprise.context.ApplicationScoped;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The type Failed step dao.
 */
@Slf4j
@ApplicationScoped
public class FailedStepDAO extends AbstractDAO<FailedStep> {

    /**
     * Finds the failed steps by search criteria.
     *
     * @param searchCriteria the search criteria.
     * @return the corresponding list of failed steps
     * @throws ServiceException if the method fails.
     */
    public List<FailedStep> findBySearchCriteria(FailedStepSearchCriteria searchCriteria) {
        List<FailedStep> results;

            CriteriaBuilder cb = getEntityManager().getCriteriaBuilder();
            CriteriaQuery<FailedStep> cq = cb.createQuery(FailedStep.class).distinct(true);
            Root<FailedStep> root = cq.from(FailedStep.class);
            List<Predicate> predicates = new ArrayList<>();
            if (searchCriteria != null) {
                if (searchCriteria.getProcessInstanceId() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.processInstanceId), searchCriteria.getProcessInstanceId()));
                }
                if (searchCriteria.getParentProcessInstanceId() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.parentProcessInstanceId), searchCriteria.getParentProcessInstanceId()));
                }
                if (searchCriteria.getProcessName() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.processName), searchCriteria.getProcessName()));
                }
                if (searchCriteria.getProcessStepLogGuid() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.processStepLogGuid), searchCriteria.getProcessStepLogGuid()));
                }
                if (searchCriteria.getProcessStepName() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.processStepName), searchCriteria.getProcessStepName()));
                }
                if (searchCriteria.getReferenceBid() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.referenceBid), String.valueOf(searchCriteria.getReferenceBid())));
                }
                if (searchCriteria.getReferenceKey() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.referenceKey), searchCriteria.getReferenceKey()));
                }
                if (searchCriteria.getWorkItemId() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.workItemId), searchCriteria.getWorkItemId()));
                }
                if (searchCriteria.getFailureType() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.failureType), searchCriteria.getFailureType()));
                }
                if (searchCriteria.getStatus() != null) {
                    predicates.add(cb.equal(root.get(FFailedStep_.status), searchCriteria.getStatus()));
                }
            }
            if (!predicates.isEmpty()) {
                cq.where(cb.and(predicates.toArray(new Predicate[0])));
            }
            results = getEntityManager().createQuery(cq).getResultList();

        return results;
    }

    /**
     * Gets process instance id to has failed step map by instance id list.
     *
     * @param processInstanceIds the process instance ids
     * @return the process instance id to has failed step map by instance id list
     */
    public Map<Long, Boolean> getProcessInstanceIdToHasFailedStepMapByInstanceIdList(List<Long> processInstanceIds)  {


            if (processInstanceIds == null || processInstanceIds.isEmpty()) {
                return new HashMap<>();
            }

            CriteriaBuilder cb = getEntityManager().getCriteriaBuilder();
            CriteriaQuery<Object[]> cq = cb.createQuery(Object[].class);
            Root<FailedStep> failedStepRoot = cq.from(FailedStep.class);

            cq.multiselect(failedStepRoot.get(FFailedStep_.processInstanceId), cb.count(failedStepRoot));

            cq.where(failedStepRoot.get(FFailedStep_.processInstanceId).in(processInstanceIds), cb.equal(failedStepRoot.get(FFailedStep_.status), FailedStep.FailedStepStatus.OPEN));
            cq.groupBy(failedStepRoot.get(FFailedStep_.processInstanceId));

            List<Object[]> results = getEntityManager().createQuery(cq).getResultList();
            Map<Long, Boolean> ret = new HashMap<>();
            for (Object[] result : results) {
                Long processInstanceId = (Long) result[0];
                int count = ((Number) result[1]).intValue();
                ret.put(processInstanceId, count > 0);
            }

            return ret;

    }
}
