package ext.api.centrallog.api;

import ext.api.centrallog.model.*;
import io.smallrye.reactive.messaging.amqp.OutgoingAmqpMetadata;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.reactive.messaging.Channel;
import org.eclipse.microprofile.reactive.messaging.Emitter;
import org.eclipse.microprofile.reactive.messaging.Message;
import org.eclipse.microprofile.reactive.messaging.Metadata;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import javax.validation.Valid;
import java.util.Date;


/**
 * The type Process log event emitter.
 */
@ApplicationScoped
@Slf4j
@Transactional
public class ProcessLogEventEmitter {

    /**
     * The constant EVENT_QUEUE.
     */
    public static final String EVENT_QUEUE = "process-log-events";


    /**
     * The Emitter.
     */
    @Inject
    @Channel("jms.queue.process-log-events")
    Emitter<String> emitter;

    /**
     * Emit process event.
     *
     * @param event the event
     * @throws RhpamException the rhpam exception
     */
    public void emitProcessEvent(@Valid NodeStartEvent event) throws RhpamException {
        this.emitEvent(event);
    }

    /**
     * Emit process event.
     *
     * @param event the event
     * @throws RhpamException the rhpam exception
     */
    public void emitProcessEvent(@Valid NodeEndEvent event) throws RhpamException {
        this.emitEvent(event);
    }

    /**
     * Emit process event.
     *
     * @param event the event
     * @throws RhpamException the rhpam exception
     */
    public void emitProcessEvent(@Valid ProcessEndEvent event) throws RhpamException {
        this.emitEvent(event);
    }

    /**
     * Emit process event.
     *
     * @param event the event
     * @throws RhpamException the rhpam exception
     */
    public void emitProcessEvent(@Valid ProcessStartEvent event) throws RhpamException {
        this.emitEvent(event);
    }

    /**
     * Emit process event.
     *
     * @param event the event
     * @throws RhpamException the rhpam exception
     */
    public void emitProcessEvent(@Valid OpenIncidentEvent event) throws RhpamException {
        this.emitEvent(event);
    }


    private void emitEvent(@Valid ProcessLogEvent event) throws RhpamException {
        try {
            log.info("Central log event: {}", event.toPrettyString());
            OutgoingAmqpMetadata meta = OutgoingAmqpMetadata.builder()
                    .withCreationTime(new Date().getTime())
                    .withApplicationProperty("EVENT_TYPE", event.getProcessEventType().name())
                    .build();
            String eventPayload = CentralLogObjectMapper.writer.writeValueAsString(event);
            this.emitter.send(Message.of(eventPayload, Metadata.of(meta)));
        } catch (Exception e) {
            throw new RhpamException(ErrorCodes.FAILED_TO_EMIT_MSG_EVENT.name(), e);
        }

    }

    /**
     * Emit process event.
     *
     * @param event the event
     * @throws RhpamException the rhpam exception
     */
    public void emitProcessEvent(ResolvedIncidentEvent event) throws RhpamException {
        this.emitEvent(event);
    }


    /**
     * The enum Error codes.
     */
    enum ErrorCodes {
        /**
         * Failed to emit msg event error codes.
         */
        FAILED_TO_EMIT_MSG_EVENT;
    }
}
