package org.tkit.rhpam.quarkus.messaging;

import com.fasterxml.jackson.core.JsonProcessingException;
import ext.api.centrallog.api.ProcessLogEventEmitter;
import ext.api.centrallog.api.ProcessMessageEventEmitter;
import ext.api.centrallog.model.*;
import io.opentracing.Tracer;
import io.opentracing.util.GlobalTracer;
import io.smallrye.reactive.messaging.amqp.AmqpMessage;
import io.smallrye.reactive.messaging.amqp.IncomingAmqpMetadata;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.config.inject.ConfigProperty;
import org.eclipse.microprofile.reactive.messaging.Message;
import org.slf4j.MDC;
import org.tkit.rhpam.quarkus.common.Config;
import org.tkit.rhpam.quarkus.emitters.FailedStepEmitter;
import org.tkit.rhpam.quarkus.emitters.JbpmMessageEmitter;
import org.tkit.rhpam.quarkus.messaging.common.ExceptionUtil;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.messaging.model.AdditionalErrorInfo;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecution;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecutionResult;
import org.tkit.rhpam.quarkus.processlog.domain.daos.DomainProcessInfoDAO;
import org.tkit.rhpam.quarkus.processlog.domain.models.DomainProcessInfo;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import static org.tkit.rhpam.quarkus.messaging.common.MessageUtil.*;

/**
 * The process client service implementation.
 */
@Slf4j
@ApplicationScoped
@Transactional()
public class ProcessClientServiceV2 {

    /**
     * The constant TKIT_RHPAM_SKIP_MESSAGE.
     */
    public static final String TKIT_RHPAM_SKIP_MESSAGE = "TKIT_RHPAM_SKIP_MESSAGE";

    private static final boolean AUTO_RETRY_ENABLED_DEFAULT = false;
    private static final int AUTO_RETRY_MAX_COUNT_DEFAULT = 3;

    private static final String AUTO_RETRY_ENABLED_PARAM_KEY = "autoRetryEnabled";
    private static final String AUTO_RETRY_MAX_COUNT_PARAM_KEY = "autoRetryMaxCount";

    @ConfigProperty(name="org.tkit.rhpam.auto-retry-on-error", defaultValue = "false")
    boolean autoRetryOnErrorDefault;

    /**
     * The Msg emitter.
     */
    @Inject
    ProcessMessageEventEmitter msgEmitter;

    /**
     * The Process log event emitter.
     */
    @Inject
    ProcessLogEventEmitter processLogEventEmitter;

    /**
     * The Failed step service.
     */
    @Inject
    FailedStepService failedStepService;

    /**
     * The Failed step emitter.
     */
    @Inject
    FailedStepEmitter failedStepEmitter;

    /**
     * The Jbpm message emitter.
     */
    @Inject
    JbpmMessageEmitter jbpmMessageEmitter;

    /**
     * The DomainProcessInfo DAO service.
     */
    @Inject
    DomainProcessInfoDAO domainProcessInfoDAO;

    /**
     * End log.
     *
     * @param execution           the execution
     * @param result              the result
     * @param message             the message
     * @param exception           the exception
     * @param additionalErrorInfo the additional error info
     * @throws Exception the exception
     */
    @Transactional(Transactional.TxType.REQUIRES_NEW)
    public void endLog(ProcessStepExecution execution, ProcessStepExecutionResult result, AmqpMessage<String> message, Throwable exception,
                       AdditionalErrorInfo additionalErrorInfo) throws Exception {
        if (exception == null) {
            endSuccessLog(execution, result, message);
            //if this failed, then then a) we let the business logic fail as well, or b) we leave business logic as is and we do some recovery
        } else {
            endFailedLog(execution, message, exception, additionalErrorInfo);
        }
    }

    private void endSuccessLog(ProcessStepExecution execution, ProcessStepExecutionResult result, AmqpMessage<String> message) throws Exception {
        // serialize parameters
        emitNodeEndEvent(execution, result);
        //if business failed, create failedStep and store in DB
        if (result.getStatus() == org.tkit.rhpam.quarkus.messaging.model.ResolutionStatus.FAILED) {
            failedStepService.createOrUpdateBusinessFailedStep(execution, result);
        }
        // log succeed log item
        jbpmMessageEmitter.notifyJBPM(execution, message, result);
    }

    private void emitNodeEndEvent(ProcessStepExecution execution, ProcessStepExecutionResult result) throws RhpamException {
        NodeEndEvent endEvent = createNodeEndEvent(execution);
        endEvent.setResolution(Resolution.fromValue(result.getStatus().name()));
        endEvent.setVariables(result.getParameters());

        updateDomainProcessInfo(execution, endEvent);

        processLogEventEmitter.emitProcessEvent(endEvent);
    }

    private void emitNodeStartEvent(ProcessStepExecution execution) throws RhpamException {
        NodeStartEvent startEvent = new NodeStartEvent();
        startEvent.setNodeId(execution.getNodeId());
        startEvent.setNodeName(execution.getName());
        startEvent.setProcessId(execution.getProcessId());
        startEvent.setProcessInstanceId(String.valueOf(execution.getProcessInstanceId()));
        startEvent.setNodeType(NodeType.ACTIVITY);
        startEvent.setCorrelationId(execution.getCorrelationId());
        startEvent.setBusinessRelevant(true);
        startEvent.setEventTime(execution.getExecutionDate());
        startEvent.setProcessEventType(EventType.NODE_START_EVENT);
        startEvent.setExecutionId(execution.getExecutionId());
        startEvent.getMetadata().put("parentProcessInstanceId", execution.getParentProcessInstanceId());
        startEvent.getMetadata().put("referenceId", execution.getReferenceId());
        startEvent.getMetadata().put("referenceKey", execution.getReferenceKey());
        startEvent.setVariables(execution.getParameters());

        updateDomainProcessInfo(execution);

        processLogEventEmitter.emitProcessEvent(startEvent);
    }

    /**
     * Technical error.
     *
     * @param workItem            the work item
     * @param result              the result
     * @param message             the message
     * @param exception           the exception
     * @param additionalErrorInfo the additional error info
     */
    @Transactional(Transactional.TxType.REQUIRES_NEW)
    public void technicalError(ProcessStepExecution workItem, ProcessStepExecutionResult result, Message message, Exception exception,
                               AdditionalErrorInfo additionalErrorInfo) {
        try {

            String stacktrace = ExceptionUtil.getExceptionStackTrace(exception);
            String errorCode = null;
            if (additionalErrorInfo != null) {
                errorCode = additionalErrorInfo.getErrorCode();
            }

//            if (exception instanceof LocalizableException) {
            //TODO tkit excpetion missing
//                LocalizableException ee = (LocalizableException) exception;
//                if (errorCode == null) {
//                    //if no specific error code is specified
//                    errorCode = "" + ee.getMessageKey();
//                }
//            }
            if (errorCode == null) {
                //fallback code
                errorCode = "TKIT_RHPAM_1000";
            }

            Map<String, Object> messageBody = new HashMap<>();
            messageBody.put("body", workItem.getParameters());
//            messageBody.put("processStepExecutionLogGuid", workItem.getProcessStepExecutionLogGuid());
//            messageBody.put("processStepLogGuid", workItem.getProcessStepLogGuid());
            messageBody.put("result", result);
            messageBody.put("stacktrace", stacktrace);

            failedStepEmitter.sendToFailedStepQueue(message, messageBody, additionalErrorInfo, errorCode);

        } catch (Exception cex) {
            // this is the last change to save information
            log.error("Error complete the message.", cex);
        }
    }

    private void endFailedLog(ProcessStepExecution execution, Message message, Throwable ex, AdditionalErrorInfo additionalErrorInfo) throws RhpamException, IOException {

        String errorCode = getErrorCode(ex, additionalErrorInfo);

        // emit event for central process log
        emitNodeEndEvent(execution, ex, additionalErrorInfo);
        // emit msg event with exception data for central process log
        emitMessageEvent(execution, errorCode, ex, additionalErrorInfo);

        // either retry or move to incident queue
        if (isAutoRetryAllowed(execution, message)) {
            log.info("Retry message for {} by sending nack", execution.getName());
            // if the message should be retried, throw exception
            throw new RhpamException("Error executing the work item! Retrying...");
        } else {
            log.info("Retry not allowed for {}, send to failed step", execution.getName());
            // if the message should not be retried, send the message to the failed steps queue
            failedStepEmitter.sendToFailedStepQueue(message, null, additionalErrorInfo, errorCode);
        }
    }

    private void emitMessageEvent(ProcessStepExecution execution, String errorCode, Throwable ex, AdditionalErrorInfo additionalErrorInfo) throws JsonProcessingException {
        boolean skipMessage = getShouldSkipMessage(ex);

        String stacktrace = ExceptionUtil.getExceptionStackTrace(ex);
        if (!skipMessage) {
            MessageEvent msg = MessageFactory.createMessage(execution, errorCode, Severity.ERROR);

            msg.setContent(stacktrace);
            msgEmitter.emitProcessMessageEvent(msg);
        }
    }

    private void emitNodeEndEvent(ProcessStepExecution execution, Throwable ex, AdditionalErrorInfo additionalErrorInfo) throws RhpamException {
        NodeEndEvent endEvent = createNodeEndEvent(execution);

        endEvent.setResolution(Resolution.FAILED);
        Map<String, Object> variables = new HashMap<>();
        ///add error info in variables
        variables.put("exception", ex.getClass().getName());
        endEvent.setVariables(variables);

        updateDomainProcessInfo(execution, endEvent);

        processLogEventEmitter.emitProcessEvent(endEvent);
    }

    private void updateDomainProcessInfo(ProcessStepExecution execution, NodeEndEvent event) {
        //TODO parentProcessInstanceId == parentProcessId?
        if (execution.getParentProcessInstanceId() == null) {
            //TODO referenceId == referenceBid
            DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(execution.getReferenceKey(), String.valueOf(execution.getReferenceId()));
            if (dpi != null) {

                dpi.setCurrentProcessInstanceId(execution.getProcessInstanceId());
                dpi.setCurrentProcessStepName(execution.getName());
                if (event.getResolution() == Resolution.FAILED) {
                    dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.ERROR);
                    dpi.setCurrentProcessStepStatus(DomainProcessInfo.ProcessStatus.ERROR.name());
                } else {
                    if (dpi.getCurrentProcessStatus() == DomainProcessInfo.ProcessStatus.ERROR) {
                        dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.RUNNING);
                    }
                    dpi.setCurrentProcessStepStatus(DomainProcessInfo.ProcessStatus.COMPLETED.name());
                }
                domainProcessInfoDAO.update(dpi);
            }
        }
    }

    private void updateDomainProcessInfo(ProcessStepExecution execution) {
        DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(execution.getReferenceKey(), String.valueOf(execution.getReferenceId()));
        //TODO parentProcessInstanceId == parentProcessId?
        if (dpi != null && execution.getParentProcessInstanceId() == null) {
            //TODO referenceId == referenceBid
            //only update if we are a step of the top level process
            dpi.setCurrentProcessInstanceId(execution.getProcessInstanceId());
            dpi.setCurrentProcessStepName(execution.getName());
            if (dpi.getCurrentProcessStatus() == DomainProcessInfo.ProcessStatus.ERROR || dpi.getCurrentProcessStatus() == DomainProcessInfo.ProcessStatus.PENDING) {
                dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.RUNNING);
            }
            dpi.setCurrentProcessStepStatus(DomainProcessInfo.ProcessStatus.PENDING.name());
            domainProcessInfoDAO.update(dpi);
        }
    }

    private NodeEndEvent createNodeEndEvent(ProcessStepExecution execution) {
        NodeEndEvent endEvent = new NodeEndEvent();
        endEvent.setNodeId(execution.getNodeId());
        endEvent.setProcessId(execution.getProcessId());
        endEvent.setProcessInstanceId(String.valueOf(execution.getProcessInstanceId()));

        endEvent.setBusinessRelevant(true);

        endEvent.setCorrelationId(execution.getCorrelationId());
        endEvent.setEventTime(new Date());
        endEvent.setExecutionId(execution.getExecutionId());
        endEvent.getMetadata().put("parentProcessInstanceId", execution.getParentProcessInstanceId());
        // for us, this is the ExecutionLog finish, the step as such is finished, when Jbpm confirms it (ProcessLogMDB...)
        endEvent.getMetadata().put("TKIT_DO_NOT_CLOSE_STEP", true);
        endEvent.setProcessEventType(EventType.NODE_END_EVENT);
        return endEvent;
    }

    private boolean getShouldSkipMessage(Throwable ex) {
        //TODO not in quarkus?
//        if (ex instanceof LocalizableException) {
//            LocalizableException ee = (LocalizableException) ex;
//
//            if (ee.getNamedParameters().containsKey(TKIT_RHPAM_SKIP_MESSAGE)) {
//                return (boolean) ee.getNamedParameters().get(TKIT_RHPAM_SKIP_MESSAGE);
//            }
//        }
        return false;
    }

    private String getErrorCode(Throwable ex, AdditionalErrorInfo additionalErrorInfo) {
        if (additionalErrorInfo != null) {
            return additionalErrorInfo.getErrorCode();
        }

//        if (ex instanceof LocalizableException) {
//            LocalizableException ee = (LocalizableException) ex;
//            if (ee.getMessageKey() != null) {
//                return ee.getMessageKey().name();
//            }
//
//        }
        //fallback code
        return "TKIT_RHPAM_1000";
    }


    /**
     * Start log process step execution.
     *
     * @param message the message
     * @return the process step execution
     */
    @Transactional(Transactional.TxType.REQUIRES_NEW)
    public ProcessStepExecution startLog(Message message) {
        try {
            // deserialize the process work item data
            ProcessStepExecution execution = msgToStepExecution(message);

            MDC.put("TKIT_PROCESS_ID", execution.getProcessId());
            MDC.put("TKIT_PROCESS_INSTANCE_ID", String.valueOf(execution.getProcessInstanceId()));
            MDC.put("TKIT_PROCESS_STEP", String.valueOf(execution.getName()));
            MDC.put("TKIT_PROCESS_REF", execution.getReferenceId());
            MDC.put("TKIT_PROCESS_REF_TYPE", execution.getReferenceKey());
            // if there is a tracer with active span modify it's name according to current process step
            if (GlobalTracer.isRegistered()) {
                Tracer tracer = GlobalTracer.get();
                if (tracer != null && tracer.activeSpan() != null) {
                    tracer.activeSpan().setOperationName(execution.getName());
                    tracer.activeSpan().setTag("process", execution.getProcessName());
                    tracer.activeSpan().setTag("processInstance", execution.getProcessInstanceId());
                    tracer.activeSpan().setTag("processRef", execution.getReferenceId());
                    tracer.activeSpan().setTag("processRefType", execution.getReferenceKey());
                }
            }

            emitNodeStartEvent(execution);

            return execution;
        } catch (Exception ex) {
            throw new RuntimeException("Error create start log", ex);
        }
    }

    private ProcessStepExecution msgToStepExecution(Message msg) throws IOException {
        ProcessStepExecution execution = ProcessStepExecution.builder().build();
        // get the header information
        execution.setProcessInstanceId(getProperty(msg, PROP_PROCESS_INSTANCE_ID));
        execution.setWorkItemId(getProperty(msg, PROP_WORK_ITEM_ID));
        execution.setDeploymentId(getProperty(msg, PROP_DEPLOYMENT_ID));
        execution.setProcessVersion(getProperty(msg, PROP_PROCESS_VERSION));
        execution.setProcessId(getProperty(msg, PROP_PROCESS_ID));
        execution.setParentProcessInstanceId(getProperty(msg, PROP_PARENT_PROCESS_INSTANCE_ID));
        Long refBid = getProperty(msg, PROP_REFERENCE_BID);
        execution.setReferenceId("" + refBid);
        execution.setReferenceKey(getProperty(msg, PROP_REFERENCE_KEY));
        execution.setProcessName(getProperty(msg, PROP_PROCESS_NAME));
        Long nodeId = getProperty(msg, PROP_KIE_NODE_ID);
        execution.setNodeId(nodeId != null ? String.valueOf(nodeId) : null);
        execution.setNodeType(org.tkit.rhpam.quarkus.messaging.model.NodeType.WORK_ITEM);
        execution.setCorrelationId(UUID.randomUUID().toString());
        Long workItemId = getProperty(msg, PROP_WORK_ITEM_ID);
        execution.setExecutionId("" + workItemId);
        execution.setName(getProperty(msg, PROP_KIE_NODE_NAME));
        execution.setExecutionDate(getStringDateProperty(msg, PROP_KIE_EXECUTION_DATE, new Date()));
        // deserialize the parameters from body
        execution.setParameters(deserializeBody(msg));

        return execution;

    }


    /**
     * Return true if the message should be retried, false otherwise
     *
     * @param execution
     * @param message   to read the parameters
     * @return true if message should be retried, false otherwise
     */
    private boolean isAutoRetryAllowed(ProcessStepExecution execution, Message message) {
        // default value false
        boolean retryEnabled = AUTO_RETRY_ENABLED_DEFAULT;
        int deliveryCount = -1;
        Object autoRetryEnabled = null;
        Object autoRetryMaxCount = null;
        try {
            IncomingAmqpMetadata meta = (IncomingAmqpMetadata) message.getMetadata(IncomingAmqpMetadata.class).get();
            //+1 because it starts from 0
            deliveryCount = meta.getDeliveryCount() + 1;
            Map<String, Object> paramMap = execution.getParameters();
            if (paramMap != null) {
                autoRetryEnabled = paramMap.get(AUTO_RETRY_ENABLED_PARAM_KEY);
                autoRetryMaxCount = paramMap.get(AUTO_RETRY_MAX_COUNT_PARAM_KEY);
            }
        } catch (Exception e) {
            log.error("Error reading body from message", e);
        }
        // if the flag could not be read/found in the message, read environment variable
        if (autoRetryEnabled == null) {
            autoRetryEnabled = autoRetryOnErrorDefault;
        }
        if (autoRetryEnabled != null) {
            if (autoRetryEnabled instanceof String) {
                retryEnabled = Boolean.parseBoolean((String) autoRetryEnabled);
            } else if (autoRetryEnabled instanceof Boolean) {
                retryEnabled = (Boolean) autoRetryEnabled;
            } else {
                log.warn("Auto retry enabled flag cannot be parsed from {} to boolean", autoRetryEnabled);
                return false;
            }
        }
        // if the max count was not defined on the message, read environment variable
        int retryMaxCount = getAutoRetryMaxCount(autoRetryMaxCount);

        // if auto retry should be enabled and the delivered count is still lower then retry max count do the retry
        boolean result = (retryEnabled && retryMaxCount > deliveryCount);
        log.info("Auto retry {} max count {} delivery count {}. Result: {}", retryEnabled, retryMaxCount, deliveryCount, result);
        return result;
    }

    /**
     * Gets the auto retry maximum count.
     *
     * @param autoRetryMaxCount the value from the process.
     * @return the corresponding auto retry max count.
     */
    private int getAutoRetryMaxCount(Object autoRetryMaxCount) {
        // default value 3 times
        int retryMaxCount = AUTO_RETRY_MAX_COUNT_DEFAULT;
        if (autoRetryMaxCount == null) {
            autoRetryMaxCount = System.getenv(Config.AUTO_RETRY_MAX_COUNT_ENV_KEY);
        }
        if (autoRetryMaxCount != null) {
            if (autoRetryMaxCount instanceof String) {
                try {
                    retryMaxCount = Integer.parseInt((String) autoRetryMaxCount);
                } catch (Exception e) {
                    log.warn("Could not parse max retry count string {} to number", autoRetryMaxCount);
                }
            } else if (autoRetryMaxCount instanceof Integer) {
                retryMaxCount = (Integer) autoRetryMaxCount;
            } else {
                log.warn("Max retry count is of undefined type with value {}. Using default.", autoRetryMaxCount);
            }
        }
        return retryMaxCount;
    }

}
