package org.tkit.rhpam.quarkus.process;

import io.smallrye.reactive.messaging.amqp.OutgoingAmqpMetadata;
import io.vertx.core.json.JsonObject;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.reactive.messaging.Channel;
import org.eclipse.microprofile.reactive.messaging.Emitter;
import org.eclipse.microprofile.reactive.messaging.Message;
import org.tkit.rhpam.quarkus.domain.daos.FailedStepDAO;
import org.tkit.rhpam.quarkus.domain.models.FailedStep;
import org.tkit.rhpam.quarkus.domain.models.FailedStep.FailedStepStatus;
import org.tkit.rhpam.quarkus.domain.models.FailedStep.FailureType;
import org.tkit.rhpam.quarkus.messaging.common.MessageUtil;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Service to retry a step (will reconstruct a messages from the FailedStep to the original queue)
 */
@Slf4j
@ApplicationScoped
@Transactional
public class RetryStepService {

    @Inject
    private FailedStepDAO failedStepDAO;

    @Inject
    @Channel("tkitRhpamWih")
    Emitter<String> emitter;

    @Inject
    private ProcessService processService;

    public void resolveFailedStep(String failedStepId, String resolutionOption, String source) throws Exception {
        FailedStep failedStep = failedStepDAO.findById(failedStepId);
        if (failedStep == null) {
            throw new RhpamException(RetryServiceErrorKeys.FAILED_STEP_NOT_FOUND.name());
        }
        if (resolutionOption == null) {
            throw new RhpamException(RetryServiceErrorKeys.MISSING_RESOLUTION_OPTION.name());
        }

        Set<String> allowedResolutionOptions = new HashSet<>();
        if (failedStep.getResolutionOptionsString() != null) {
            String[] optionArr = failedStep.getResolutionOptionsString().split(",");
            for (String item : optionArr) {
                allowedResolutionOptions.add(item.trim().toLowerCase());
            }
        }

        resolutionOption = resolutionOption.trim();

        if (!allowedResolutionOptions.contains(resolutionOption.toLowerCase())) {
            log.error("Invalid resolution option received {}", resolutionOption);
            StringBuilder sb = new StringBuilder();
            sb.append(RetryServiceErrorKeys.INVALID_RESOLUTION_OPTIONS.name());
            sb.append(" option: '");
            sb.append(resolutionOption);
            sb.append("' is not one of allowed options: ");
            sb.append(allowedResolutionOptions);
            throw new RhpamException(sb.toString());
        }

        if (failedStep.getFailureType() == FailureType.BUSINESS_ERROR) {
            handleBusinessErrorResolution(failedStep, resolutionOption, source);
        } else {
            handleTechnicalErrorResolution(failedStep, resolutionOption, source);
        }
    }

    private void handleBusinessErrorResolution(FailedStep failedStep, String resolutionOption, String source) throws RhpamException {
        try {
            String signalTarget = failedStep.getResolutionSignalReceiver();
            processService.sendMessage(failedStep.getDeploymentId(), failedStep.getProcessInstanceId(), signalTarget, resolutionOption);
            updateFailedStepAndMessages(failedStep, resolutionOption, source);
        } catch (Exception e) {
            throw new RhpamException(RetryServiceErrorKeys.FAILED_TO_PROCESS_BUSINESS_ERR_RESOLUTION.name(), e);
        }
    }

    private void handleTechnicalErrorResolution(FailedStep failedStep, String resolutionOption, String source) throws RhpamException {
      try {
          emitter.send(recreateMessageFromFailedStep(failedStep));
            // update execution count and status of FailedStep object
            failedStep.setExecutionCount(failedStep.getExecutionCount() + 1);
            updateFailedStepAndMessages(failedStep, resolutionOption, source);

        } catch (Exception e) {
            throw new RhpamException(RetryServiceErrorKeys.FAILED_TO_PROCESS_FATAL_ERR_RESOLUTION.name(), e);
        }
    }

    /**
     * Update the failed step to status closed and historize all message related to the process step
     *
     * @param failedStep       failed step
     * @param resolutionOption resolution option
     * @param source           source
     * @throws RhpamException on error
     */
    private void updateFailedStepAndMessages(FailedStep failedStep, String resolutionOption, String source) throws RhpamException {
        failedStep.setStatus(FailedStepStatus.CLOSED);
        failedStep.setResolution(resolutionOption);
        failedStep.setResolutionOrigin(source);
        failedStepDAO.update(failedStep);
        //TODO send ResolveIncidentEvent
    }

    /**
     * Creates the message for the failed step.
     *
     * @param failedStep the failed step message.
     * @return the created message.
     */
    private Message<String> recreateMessageFromFailedStep(FailedStep failedStep) {
        Message<String> recoveryMessage = Message.of(failedStep.getOriginalJMSMessage());

        Map<String, Object> headersMap = MessageUtil.deserialize(failedStep.getHeaders());
        OutgoingAmqpMetadata meta = OutgoingAmqpMetadata.builder()
                .withApplicationProperties(JsonObject.mapFrom(headersMap))
                .build();
        recoveryMessage.addMetadata(meta);

        return recoveryMessage;
    }


    enum RetryServiceErrorKeys {
        FAILED_STEP_NOT_FOUND,
        MISSING_RESOLUTION_OPTION,
        INVALID_RESOLUTION_OPTIONS,
        FAILED_TO_PROCESS_BUSINESS_ERR_RESOLUTION,
        FAILED_TO_PROCESS_FATAL_ERR_RESOLUTION
    }
}
