package org.tkit.rhpam.quarkus.messaging;

import io.smallrye.reactive.messaging.amqp.AmqpMessage;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.context.ThreadContext;
import org.eclipse.microprofile.reactive.messaging.Acknowledgment;
import org.eclipse.microprofile.reactive.messaging.Incoming;
import org.eclipse.microprofile.reactive.messaging.Message;
import org.tkit.rhpam.quarkus.domain.daos.DomainProcessInfoDAO;
import org.tkit.rhpam.quarkus.domain.models.DomainProcessInfo;
import org.tkit.rhpam.quarkus.domain.models.FailedStep;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;

import javax.inject.Inject;
import javax.transaction.Transactional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

@Slf4j
@Transactional
public class TechnicalErrorFailedStepListener {

    @Inject
    private FailedStepService service;


    @Inject
    ThreadContext threadContext;

//    @Inject
//    private SupportToolSubmitterService supportToolSubmitterService;

    @Inject
    private DomainProcessInfoDAO domainProcessInfoDAO;


    /**
     * Technical error handler
     *
     * @param message the message
     * @return the completion stage
     */
    @Transactional(Transactional.TxType.REQUIRED)
    @Acknowledgment(Acknowledgment.Strategy.MANUAL)
    @Incoming("tkitRhpamFailedSteps")
    //This cant have AmqpMessage as param because it technicaly might not go via AMQ
    public CompletionStage<Void> onFailedStepMessage(Message<String> message) {
        return threadContext.withContextCapture(CompletableFuture.runAsync(() -> {
            try {
                //store in DB
                FailedStep failedStep = service.createOrUpdateFailedStep(message);
                //send to central log
                service.emitFailedStepEvent(failedStep);
                DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(failedStep.getReferenceKey(),
                        String.valueOf(failedStep.getReferenceBid()));
                if (dpi != null) {
                    //only update if we are a step of the top level process
                    dpi.setCurrentProcessStepStatus(DomainProcessInfo.ProcessStatus.ERROR.name());
                    dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.ERROR);
                    dpi.setCurrentProcessStepName(failedStep.getProcessStepName());
                    domainProcessInfoDAO.update(dpi);
                }
                //optionally notify support tool
//            supportToolSubmitterService.submitExternalSupportToolIssue(failedStep);
                message.ack();
            } catch (Exception e) {
                message.nack(e);
                throw new IllegalArgumentException("Error saving/updating failed step to database.", e);
            }
        }));
    }
}
