/*
 * Copyright (c) 2018, 1000kit.org, and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.tkit.rhpam.quarkus.process;

import io.quarkus.security.identity.SecurityIdentity;
import io.smallrye.reactive.messaging.amqp.OutgoingAmqpMetadata;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.reactive.messaging.Channel;
import org.eclipse.microprofile.reactive.messaging.Emitter;
import org.eclipse.microprofile.reactive.messaging.Message;
import org.eclipse.microprofile.reactive.messaging.Metadata;
import org.tkit.rhpam.quarkus.domain.daos.DomainProcessInfoDAO;
import org.tkit.rhpam.quarkus.domain.models.DomainProcessInfo;
import org.tkit.rhpam.quarkus.messaging.common.MessageUtil;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.messaging.model.CommandType;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import static org.tkit.rhpam.quarkus.messaging.common.MessageUtil.*;

/**
 * The process service.
 */
@Slf4j
@ApplicationScoped
@Transactional(Transactional.TxType.REQUIRED)
public class ProcessService {

    @Inject
    SecurityIdentity identity;
    @Inject
    DomainProcessInfoDAO domainProcessInfoDAO;
    /**
     * The Emitter.
     */
    @Inject
    @Channel(RESPONSE_QUEUE)
    Emitter<String> emitter;

    /**
     * Starts the process.
     *
     * @param deploymentId the deployment ID.
     * @param processId    the ID of the process
     * @param referenceBid business reference bid to be defined in the step logs
     * @param referenceKey the reference key.
     * @param data         the input data for the process.
     * @return the process log GUID.
     * @throws RhpamException if the method fails. {@link ErrorKeys#ERROR_START_PROCESS}
     */
    public String startProcess(String deploymentId, String processId, Long referenceBid, String referenceKey, Map<String, Object> data) throws RhpamException {

        try {
            // create process start log
            String parameters = MessageUtil.serializeBody(data);

            updateDomainProcessInfo(processId, referenceBid, referenceKey);

            String guid = UUID.randomUUID().toString();

            OutgoingAmqpMetadata meta = OutgoingAmqpMetadata.builder()
                    .withMessageId(UUID.randomUUID().toString())
                    .withCreationTime(new Date().getTime())
//                    .withMessageAnnotations("x-opt-jms-msg-type", 5)
                    .withApplicationProperty(PROP_DEPLOYMENT_ID, deploymentId)
                    .withApplicationProperty(PROP_PROCESS_LOG_GUID, guid)
                    .withApplicationProperty(PROP_REFERENCE_BID, referenceBid)
                    .withApplicationProperty(PROP_REFERENCE_KEY, referenceKey)
                    .withApplicationProperty(PROP_PROCESS_ID, processId)
                    .withApplicationProperty(PROP_CMD, CommandType.START_PROCESS.name()).build();
            Message<String> toEmit = Message.of(parameters, Metadata.of(meta),  () -> {
                // Called when the message is acknowledged.
                log.info("Message ack");
                return CompletableFuture.completedFuture(null);
            });
            //TODO tracing
            emitter.send(toEmit);

            return guid;
        } catch (Exception ex) {
            throw new RhpamException(ErrorKeys.ERROR_START_PROCESS.name(), ex);
        }
    }

    /**
     * Sends the message to the process.
     *
     * @param deploymentId      the deployment ID.
     * @param processInstanceId the process instance ID.
     * @param messageId         the message ID.
     * @param data              the input data.
     * @throws RhpamException if the method fails. {@link ErrorKeys#ERROR_SEND_SIGNAL}
     */
    public void sendMessage(String deploymentId, Long processInstanceId, String messageId, Object data) throws RhpamException {
        sendSignal(deploymentId, processInstanceId, "Message-" + messageId, data);
    }

    /**
     * Sends the signal to the process.
     *
     * @param deploymentId      the deployment ID.
     * @param processInstanceId the process instance ID.
     * @param signalId          the signal ID.
     * @param data              the input data.
     * @throws RhpamException if the method fails. {@link ErrorKeys#ERROR_SEND_SIGNAL}
     */
    public void sendSignal(String deploymentId, Long processInstanceId, String signalId, Object data) throws RhpamException {

        try {
            Map<String, Object> tmp = new HashMap<>();
            if (data != null) {
                tmp.put(MessageUtil.PROP_SIGNAL_DATA, data);
            }
            String parameters = MessageUtil.serializeBody(tmp);

            Message<String> toEmit = Message.of(parameters);
            OutgoingAmqpMetadata meta = OutgoingAmqpMetadata.builder()
                    .withApplicationProperty(PROP_DEPLOYMENT_ID, deploymentId)
                    .withApplicationProperty(PROP_PROCESS_INSTANCE_ID, processInstanceId)
                    .withApplicationProperty(PROP_SIGNAL_ID, signalId)
                    .withApplicationProperty(PROP_CMD, CommandType.SEND_SIGNAL.name())
                    .build();
            toEmit = toEmit.addMetadata(meta);
            //TODO tracing
            emitter.send(toEmit);

        } catch (Exception ex) {
            throw new RhpamException(ErrorKeys.ERROR_SEND_SIGNAL.name(), ex);
        }
    }

    private String getPrincipal() {
        return this.identity != null && this.identity.getPrincipal() == null ? this.identity.getPrincipal().getName() : null;
    }


    private void updateDomainProcessInfo(String processId, Long referenceBid, String referenceKey) {
        DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(referenceKey, String.valueOf(referenceBid));
        if (dpi == null) {
            dpi = new DomainProcessInfo(referenceKey, String.valueOf(referenceBid));
        }
        dpi.setCurrentProcessId(processId);
        dpi.setCurrentProcessInitiator(getPrincipal());
        dpi.setCurrentProcessStartTime(new Date());
        dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.PENDING);
        dpi.setProcessActive(true);
        dpi.setCurrentProcessStepName(null);
        dpi.setCurrentProcessStepStatus(null);

        domainProcessInfoDAO.update(dpi);
    }


    enum ErrorKeys {

        ERROR_START_PROCESS,

        ERROR_SEND_SIGNAL
    }
}
