package org.tkit.rhpam.quarkus.messaging;

import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.Tracer;
import io.opentracing.tag.Tags;
import io.quarkus.arc.Arc;
import io.smallrye.reactive.messaging.amqp.AmqpMessage;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.context.ManagedExecutor;
import org.eclipse.microprofile.context.ThreadContext;
import org.eclipse.microprofile.opentracing.Traced;
import org.eclipse.microprofile.reactive.messaging.Acknowledgment;
import org.eclipse.microprofile.reactive.messaging.Incoming;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.messaging.model.AdditionalErrorInfo;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecution;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecutionResult;
import org.tkit.rhpam.quarkus.process.BusinessProcess;
import org.tkit.rhpam.quarkus.process.ProcessStepService;
import org.tkit.rhpam.quarkus.tracing.TracingUtils;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

/**
 * The type Tkit rhpam wih service.
 */
@ApplicationScoped
@Slf4j
@Traced
public class TkitRhpamWIHService {

    /**
     * The Process client service v 2.
     */
    @Inject
    ProcessClientServiceV2 processClientServiceV2;

    @Inject
    ThreadContext threadContext;
    @Inject
    ManagedExecutor managedExecutor;


    @Inject
    Tracer configuredTracer;

    @Inject
    BusinessErrorHandlerStepService businessErrorHandlerStepService;


    /**
     * Test 2 completion stage.
     *
     * @param message the message
     * @return the completion stage
     */
    @Transactional(Transactional.TxType.REQUIRED)
    @Acknowledgment(Acknowledgment.Strategy.MANUAL)
    @Incoming("tkitRhpamWih")
    public CompletionStage<Void> onWIHMessage(AmqpMessage<String> message) {
        Span span = TracingUtils.buildChildSpan(message.getApplicationProperties(), configuredTracer);
        span.setOperationName("executeStepMessage");

        // TX 1 start
        if (log.isDebugEnabled()) {
            log.debug("Message properties: {}" ,message.getApplicationProperties());
        }
        return threadContext.withContextCapture(CompletableFuture.runAsync(() -> {
            try (Scope scope = configuredTracer.scopeManager().activate(span, false)) {
                // TX 2 start, TX 1 pause
                // if we fail here, whole execution fails. JMS message is returned to queue
                ProcessStepExecution execution;
                try {
                    execution = processClientServiceV2.startLog(message);
                } catch (Exception e) {
                    message.nack(e);
                    return;
                }
                // TX 2 end
                long startTime = System.currentTimeMillis();
                log.info("Process work item {} started. Attempt: {}", execution, message.getDeliveryCount()+1);
                Arc.container().requestContext().activate();
                ProcessStepService stepService =  Arc.container().instance(ProcessStepService.class, new BusinessProcess.Literal(execution.getProcessId())).get();
                Arc.container().requestContext().deactivate();
                if (stepService == null) {
                    log.error("No ProcessStepService for {}", execution.getProcessId());
                    message.nack(new IllegalArgumentException("No ProcessStepService for " + execution.getProcessId()));
                    return;
                }

                try {
                    ProcessStepExecutionResult result = stepService.doExecute(execution);
                    handleStepSuccess(execution, result, message);
                    message.ack();
                } catch (Exception e) {
                    AdditionalErrorInfo additionalErrorInfo = stepService.provideErrorInformation(execution, e);
                    log.error("Business execution failed for step: {} with error: ", execution.getName(), e.getMessage());
                    handleStepFailed(execution, e, message, additionalErrorInfo);
                }
            }

        }).thenRun(()-> {
            span.finish();
        }));

    }

    /**
     * Test 2 completion stage.
     *
     * @param message the message
     * @return the completion stage
     */
    @Transactional(Transactional.TxType.REQUIRES_NEW)
    @Acknowledgment(Acknowledgment.Strategy.MANUAL)
    @Incoming("businessErrorWih")
    public CompletionStage<Void> onBusinessErrorWIHMessage(AmqpMessage<String> message) {
        Span span = TracingUtils.buildChildSpan(message.getApplicationProperties(), configuredTracer);
        // TX 1 start
        return threadContext.withContextCapture(CompletableFuture.runAsync(() -> {
            try (Scope scope = configuredTracer.scopeManager().activate(span, false)) {
                // TX 2 start, TX 1 pause
                // if we fail here, whole execution fails. JMS message is returned to queue
                ProcessStepExecution execution;
                try {
                    execution = processClientServiceV2.startLog(message);
                } catch (Exception e) {
                    message.nack(e);
                    return;
                }
                // TX 2 end
                long startTime = System.currentTimeMillis();
                log.info("Process work item {} started. Attempt: {}", execution, message.getDeliveryCount() + 1);
                ProcessStepService stepService = businessErrorHandlerStepService;

                if (stepService == null) {
                    log.error("No ProcessStepService for {}", execution.getProcessId());
                    message.nack(new IllegalArgumentException("No ProcessStepService for " + execution.getProcessId()));
                    return;
                }

                try {
                    ProcessStepExecutionResult result = stepService.doExecute(execution);
                    handleStepSuccess(execution, result, message);
                    message.ack();
                } catch (Exception e) {
                    AdditionalErrorInfo additionalErrorInfo = stepService.provideErrorInformation(execution, e);
                    log.error("Business execution failed for step: {}", execution.getName(), e);
                    handleStepFailed(execution, e, message, additionalErrorInfo);
                }
            }
        }).thenRun(()->{
            configuredTracer.scopeManager().activate(span, true);
        }));
    }

    private void handleStepFailed(ProcessStepExecution execution, Throwable t, AmqpMessage<String> message, AdditionalErrorInfo additionalErrorInfo) {
        log.info("#### handleStepFailed: {}", t.getMessage());
        try {
            processClientServiceV2.endLog(execution, null, message, t, additionalErrorInfo);
            //endLog will throw en exception if the message should be retried(nacked)
            //if it does not throw exception, then it create a new technical incident(new msg in FailedStepQueue)
            //and we then ack the orig message, because we want to pause the process now
            message.ack();
        }  catch (RhpamException rhe) {
            message.nack(rhe);
        } catch (Exception e) {
            //we get unexpected error, but we wanted to retry the msg anyhow
            //this would most probably be a bug in endlog, or hard technical error
            log.error("Error in endlog", e);
            message.nack(e);
        }

        log.info("#### FINISHED handleStepFailed");
    }

    private void handleStepSuccess(ProcessStepExecution execution, ProcessStepExecutionResult processStepExecutionResult, AmqpMessage<String> message) {
        log.info("#### handleStepSuccess {}", processStepExecutionResult);
        try {
            processClientServiceV2.endLog(execution, processStepExecutionResult, message, null, null);
        } catch (Exception e) {
            log.error("Error in endlog", e);
        }
        log.info("### FINISH handleStepSuccess");
    }
}