package org.tkit.rhpam.quarkus.tracing;

import io.opentracing.References;
import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.propagation.Format;
import io.opentracing.propagation.Format.Builtin;
import io.opentracing.tag.Tags;
import io.vertx.core.json.JsonObject;
import org.apache.qpid.proton.amqp.messaging.ApplicationProperties;

import java.util.HashMap;
import java.util.Map;

public class TracingUtils {

    public static SpanContext extract(JsonObject props, Tracer tracer) {
        SpanContext spanContext = tracer
                .extract(Format.Builtin.TEXT_MAP, new HeadersMapExtractAdapter(props.getMap()));
        if (spanContext != null) {
            return spanContext;
        }

        Span span = tracer.activeSpan();
        if (span != null) {
            return span.context();
        }
        return null;
    }

    public static void buildAndFinishChildSpan(JsonObject props, String queue,
                                               Tracer tracer) {
        Span child = buildChildSpan(props, tracer);
        if (child != null) {
            child.finish();
        }
    }

    public static Span buildChildSpan(JsonObject props, Tracer tracer) {
        Tracer.SpanBuilder spanBuilder = tracer.buildSpan("receive")
                .withTag(Tags.SPAN_KIND.getKey(), Tags.SPAN_KIND_CONSUMER);

//        spanBuilder.ignoreActiveSpan();
        SpanContext parentContext = TracingUtils.extract(props, tracer);
        if (parentContext != null) {
            spanBuilder.addReference(References.FOLLOWS_FROM, parentContext);
        }

        Span span = spanBuilder.start();

        try {
            if (props.getMap() != null) {
                tracer.inject(span.context(), Builtin.TEXT_MAP,
                        new HeadersMapInjectAdapter(props.getMap()));
            }
        } catch (Exception e) {
            // Ignore. Headers can be immutable. Waiting for a proper fix.
        }

        return span;
    }

    public static Span buildSpan(String exchange, String routingKey, ApplicationProperties props,
                                 Tracer tracer) {
        Tracer.SpanBuilder spanBuilder = tracer.buildSpan("send")
                .ignoreActiveSpan()
                .withTag(Tags.SPAN_KIND.getKey(), Tags.SPAN_KIND_PRODUCER)
                .withTag("routingKey", routingKey);

        SpanContext spanContext = null;

        if (props != null && props.getValue() != null) {
            // just in case if span context was injected manually to props in basicPublish
            spanContext = tracer.extract(Format.Builtin.TEXT_MAP,
                    new HeadersMapExtractAdapter(props.getValue()));
        }

        if (spanContext == null) {
            Span parentSpan = tracer.activeSpan();
            if (parentSpan != null) {
                spanContext = parentSpan.context();
            }
        }

        if (spanContext != null) {
            spanBuilder.asChildOf(spanContext);
        }

        Span span = spanBuilder.start();

        return span;
    }

    /**
     * Store current tracecontext as hashmap
     * @param props existing hashmap to inject to
     * @param span current span
     * @param tracer current tracer
     * @return filled hashmap
     */
    public static Map<String, Object> inject(Map<String, Object> props, Span span, Tracer tracer) {

        if (props == null) {
            props = new HashMap<>();
        }

        tracer.inject(span.context(), Format.Builtin.TEXT_MAP, new HeadersMapInjectAdapter(props));

        return props;
    }
}