package org.tkit.rhpam.quarkus.messaging;

import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.Tracer;
import io.smallrye.reactive.messaging.amqp.AmqpMessage;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.context.ManagedExecutor;
import org.eclipse.microprofile.context.ThreadContext;
import org.eclipse.microprofile.reactive.messaging.Acknowledgment;
import org.eclipse.microprofile.reactive.messaging.Incoming;
import org.tkit.rhpam.quarkus.domain.daos.DomainProcessInfoDAO;
import org.tkit.rhpam.quarkus.domain.models.DomainProcessInfo;
import org.tkit.rhpam.quarkus.domain.models.FailedStep;
import org.tkit.rhpam.quarkus.tracing.TraceFromMessage;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

@Slf4j
@Transactional
@ApplicationScoped
public class TechnicalErrorFailedStepListener {

    @Inject
    FailedStepService service;

    @Inject
    Tracer configuredTracer;
    @Inject
    ThreadContext threadContext;

    @Inject
    DomainProcessInfoDAO domainProcessInfoDAO;
    @Inject
    ManagedExecutor managedExecutor;


    /**
     * Technical error handler
     *
     * @param message the message
     * @return the completion stage
     */
    @Transactional(Transactional.TxType.REQUIRES_NEW)
    @Acknowledgment(Acknowledgment.Strategy.MANUAL)
    @Incoming("tkitRhpamFailedSteps")
    @TraceFromMessage
    public CompletionStage<Void> onFailedStepMessage(AmqpMessage<String> message) {
        Span span = configuredTracer.activeSpan().setOperationName("onFailedStepMessage");
        return threadContext.withContextCapture(CompletableFuture.runAsync(() -> {
            try (Scope scope = configuredTracer.scopeManager().activate(span, false)) {
                //store in DB
                FailedStep failedStep = service.createOrUpdateFailedStep(message);
                //send to central log
                service.emitFailedStepEvent(failedStep);
                DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(failedStep.getReferenceKey(),
                        String.valueOf(failedStep.getReferenceBid()));
                if (dpi != null) {
                    //only update if we are a step of the top level process
                    dpi.setCurrentProcessStepStatus(DomainProcessInfo.ProcessStatus.ERROR.name());
                    dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.ERROR);
                    dpi.setCurrentProcessStepName(failedStep.getProcessStepName());
                    domainProcessInfoDAO.update(dpi);
                }
                //optionally notify support tool
//            supportToolSubmitterService.submitExternalSupportToolIssue(failedStep);
                message.ack();
            } catch (Exception e) {
                message.nack(e);
            }
        }, managedExecutor));
    }
}
