/*
 * Copyright (c) 2018, 1000kit.org, and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.tkit.rhpam.quarkus.messaging.common;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import io.smallrye.reactive.messaging.amqp.IncomingAmqpMetadata;
import io.vertx.core.json.JsonObject;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.reactive.messaging.Message;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * The message properties
 */
@Slf4j
public final class MessageUtil {

    /**
     * The constant FAILED_STEP_ERROR_CODE.
     */
    public static final String FAILED_STEP_ERROR_CODE = "TKIT_FS_ERROR_CODE";
    /**
     * The constant FAILED_STEP_ERROR_MSG.
     */
    public static final String FAILED_STEP_ERROR_MSG = "TKIT_FS_ERROR_MSG";
    /**
     * The constant FAILED_STEP_DEEP_LINK.
     */
    public static final String FAILED_STEP_DEEP_LINK = "TKIT_FS_DEEP_LINK";
    /**
     * The constant PROP_PARENT_PROCESS_ID.
     */
    public static final String PROP_PARENT_PROCESS_ID = "TKIT_PARENT_PROCESS_ID";
    /**
     * The constant PROP_TKIT_SUB_PROCESS_ID.
     */
    public static final String PROP_TKIT_SUB_PROCESS_ID = "TKIT_SUB_PROCESS_ID";


    /**
     * The default private constructor
     */
    private MessageUtil() {
        // empty constructor
    }

    /**
     * The JMS command name property name.
     */
    public static final String PROP_CMD = "TKIT_CMD";

    /**
     * The response queue.
     */
    public static final String RESPONSE_QUEUE = "jms.queue.ResponseExecutionQueue";

    /**
     * The Failed steps queue.
     */
    public static final String ERROR_QUEUE = "jms.queue.ErrorProcessQueue";

    /**
     * The Failed steps queue.
     */
    public static final String FAILED_STEPS_QUEUE = "jms.queue.FailedStepsQueue";

    /**
     * The jms redelivery count key.
     */
    public static final String REDELIVERY_COUNT_KEY = "JMSXDeliveryCount";

    /**
     * The JMS process instance ID property name.
     */
    public static final String PROP_PROCESS_INSTANCE_ID = "TKIT_PROCESS_INSTANCE_ID";

    /**
     * The JMS deployment ID property name.
     */
    public static final String PROP_DEPLOYMENT_ID = "TKIT_DEPLOYMENT_ID";

    /**
     * The JMS process ID property name.
     */
    public static final String PROP_PROCESS_ID = "TKIT_PROCESS_ID";

    /**
     * The process process name message property name.
     */
    public static final String PROP_PROCESS_NAME = "TKIT_PROCESS_NAME";

    /**
     * The JMS work item ID property name.
     */
    public static final String PROP_WORK_ITEM_ID = "TKIT_WORK_ITEM_ID";

    /**
     * The JMS signal ID property name.
     */
    public static final String PROP_SIGNAL_ID = "TKIT_SIGNAL";

    /**
     * The JMS signal data property name.
     */
    public static final String PROP_SIGNAL_DATA = "TKIT_SIGNAL_DATA";

    /**
     * The Resolution status property.
     */
    public static final String PROP_TKIT_RESOLUTION_STATUS = "TKIT_RESOLUTION_STATUS";
    /**
     * The Correlation id property.
     */
    public static final String PROP_TKIT_CORRELATION_ID = "TKIT_CORRELATION_ID";

    /**
     * The process node ID message property name.
     */
    public static final String PROP_KIE_NODE_ID = "TKIT_NODE_ID";

    /**
     * The process listener type message property name.
     */
    public static final String PROP_KIE_LISTENER_TYPE = "TKIT_LISTENER_TYPE";

    /**
     * The work item name message property name.
     */
    public static final String PROP_WORK_ITEM_NAME = "TKIT_WORK_ITEM_NAME";

    /**
     * The node name message property name.
     */
    public static final String PROP_KIE_NODE_NAME = "TKIT_NODE_NAME";

    /**
     * The node type message property name.
     */
    public static final String PROP_KIE_NODE_TYPE = "TKIT_NODE_TYPE";

    /**
     * The process execution date message property name.
     */
    public static final String PROP_KIE_EXECUTION_DATE = "TKIT_EXECUTION_DATE";

    /**
     * The process parent process instance ID message property name.
     */
    public static final String PROP_PARENT_PROCESS_INSTANCE_ID = "TKIT_PARENT_PROCESS_INSTANCE_ID";

    /**
     * The process process version message property name.
     */
    public static final String PROP_PROCESS_VERSION = "TKIT_PROCESS_VERSION";

    /**
     * The process log event message property.
     */
    public static final String PROP_KIE_PROCESS_LOG_EVENT = "TKIT_PROCESS_LOG_EVENT";

    /**
     * The process log GUID property name.
     */
    public static final String PROP_PROCESS_LOG_GUID = "TKIT_PROCESS_LOG_GUID";

    /**
     * The process execution log GUID property name.
     */
    public static final String PROP_PROCESS_EXEC_LOG_GUID = "TKIT_PROCESS_EXEC_LOG_GUID";

    /**
     * The execution id message property name.
     */
    public static final String PROP_EXECUTION_ID = "TKIT_EXECUTION_ID";

    /**
     * The reference id.
     */
    public static final String PROP_REFERENCE_BID = "TKIT_REFERENCE_BID";

    /**
     * The reference key.
     */
    public static final String PROP_REFERENCE_KEY = "TKIT_REFERENCE_KEY";

    /**
     * Key for the boundaryEventId attribute
     */
    public static final String PROP_BOUNDARY_EVENT_ID = "TKIT_BOUNDARY_EVENT_ID";

    /**
     * The JMS timer ID property name.
     */
    public static final String PROP_TIMER_ID = "TKIT_TIMER_ID";

    /**
     * The object mapper.
     */
    private static final ObjectMapper MAPPER = new ObjectMapper();

    /**
     * The object mapper reader.
     */
    private static final ObjectReader READER = MAPPER.readerFor(Map.class);

    /**
     * The object mapper reader.
     */
    private static final ObjectWriter WRITER = MAPPER.writerFor(Map.class);

    /**
     * Property for process outcome
     */
    public static final String PROP_PROCESS_OUTCOME = "PROP_PROCESS_OUTCOME";

    /**
     * Property for process error
     */
    public static final String PROP_PROCESS_ERROR = "PROP_PROCESS_ERROR";

    /**
     * Gets the message property. Check if the property exists.
     *
     * @param <T>  the property type.
     * @param msg  the message.
     * @param name the property name.
     * @return the property value.
     */
    @SuppressWarnings("unchecked")
    public static <T> T getProperty(Message msg, String name) {
        IncomingAmqpMetadata meta = (IncomingAmqpMetadata) msg.getMetadata(IncomingAmqpMetadata.class).get();
        JsonObject msgProps = meta.getProperties();


        if ( msgProps.containsKey(name)) {
            return (T) msgProps.getValue(name);
        }
        return null;
    }

    /**
     * Gets the message property. Check if the property exists.
     *
     * @param msg         the message.
     * @param name        the property name.
     * @param defaultDate the default date.
     * @return the property date.
     */
    public static Date getStringDateProperty(Message<String> msg, String name, Date defaultDate) {
        IncomingAmqpMetadata meta = (IncomingAmqpMetadata) msg.getMetadata(IncomingAmqpMetadata.class).get();
        JsonObject msgProps = meta.getProperties();

        if ( msgProps.containsKey(name)) {
           String tmp = msgProps.getString(name);

            return stringToDate(tmp, defaultDate);
        }
        return null;
    }

    /**
     * Gets the message enum property. Check if the property exists.
     *
     * @param <T>   the type parameter
     * @param msg   the message.
     * @param name  the property name.
     * @param clazz the property enum type.
     * @return the property value.
     */
    public static <T extends Enum<T>> T getEnumProperty(Message msg, String name, Class<T> clazz)  {
        String tmp = getProperty(msg, name);
        if (tmp != null && !tmp.isEmpty()) {
            try {
                return Enum.valueOf(clazz, tmp);
            } catch (Exception ex) {
                log.error("Error parsing parameter {} as enumeration {} from the value {}", name, clazz.getName(), tmp);
                throw new RuntimeException("Not valid string for the enumeration.", ex);
            }
        }
        return null;
    }

    /**
     * Serialize the data to string for the TextMessage.
     *
     * @param data the data.
     * @return the corresponding string value.
     * @throws IOException if the serialization fails.
     */
    public static String serializeBody(Map<String, Object> data) throws IOException {

        if (data == null) {
            //AMQP messages cant be empty
            data = new HashMap<>();
        }
        return WRITER.writeValueAsString(data);
    }

    /**
     * Deserialize the data to string for the database.
     *
     * @param <T>  type of object to deserialize to
     * @param data the data.
     * @return the corresponding string value or {@code null} if the data is null.
     */
    public static <T> T deserialize(String data) {
        T result = null;
        if (data != null) {
            try {
                String tmp = "{\"value\":" + data + "}";
                Map<String, Object> r = READER.with(DeserializationFeature.USE_LONG_FOR_INTS).readValue(tmp);
                result = (T) r.get("value");
            } catch (Exception ex) {
                log.error("Error deserialize the value: " + data + ". Switch to the toString fallback!", ex);
            }
        }
        return result;
    }

    /**
     * Serialize the data to string for the database.
     *
     * @param data the data.
     * @return the corresponding string value or {@code null} if the data is null.
     */
    public static String serialize(Object data) {
        String result = null;
        if (data != null) {
            try {
                result = WRITER.forType(data.getClass()).writeValueAsString(data);
            } catch (Exception ex) {
                log.error("Error serialize the value: " + data + ". Switch to the toString fallback!", ex);
                result = "" + data;
            }
        }
        return result;
    }

    /**
     * Deserialize the message to the map of data.
     *
     * @param message the message with the string content.
     * @return the corresponding map of data.
     * @throws IOException if the deserialization of the body fails.
     */
    public static Map<String, Object> deserializeBody(Message<String> message) throws IOException {
        Map<String, Object> result = new HashMap<>();
        if (message != null) {
            String body =message.getPayload();
            if (body != null && !body.isEmpty()) {
                result = READER.with(DeserializationFeature.USE_LONG_FOR_INTS).readValue(body);
            }
        }
        return result;
    }

    /**
     * Converts the string to date.
     *
     * @param value        the string value.
     * @param defaultValue the default value.
     * @return the corresponding date value.
     */
    public static Date stringToDate(String value, Date defaultValue) {
        Date result = null;
        try {
            if (value != null && !value.isEmpty()) {
                DateFormat df = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss.SSSZ");
                result = df.parse(value);
            }
        } catch (Exception ex) {
            log.warn("Error parsing the date {}. Switch to default value {}.", value, defaultValue);
            log.error("Error date parsing.", ex);
            result = defaultValue;
        }
        return result;
    }

    /**
     * Date to string string.
     *
     * @param date the date
     * @return the string
     */
    public static String dateToString(Date date) {
        if (date != null) {
            return new SimpleDateFormat("dd.MM.yyyy HH:mm:ss.SSSZ").format(date);
        } else {
            return null;
        }
    }

//    /**
//     * Copy the message header parameters to message.
//     *
//     * @param from the message.
//     * @param to   the message.
//     */
//    public static void copyMessageHeader(Message from, Message to) {
//        Enumeration keys = from.getPropertyNames();
//        while (keys.hasMoreElements()) {
//            String key = (String) keys.nextElement();
//            Object value = from.getObjectProperty(key);
//            to.setObjectProperty(key, value);
//        }
//    }
}
