package org.tkit.rhpam.quarkus.messaging;

import ext.api.centrallog.model.EventType;
import ext.api.centrallog.model.MessageEvent;
import ext.api.centrallog.model.Severity;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecution;

import java.util.Date;

/**
 * Factory class to create messages for a specific work item
 *
 * @author mhorvath
 */
public class MessageFactory {

    private MessageFactory() {
        throw new IllegalStateException("Utility class");
    }

    /**
     * Create a message associated with the workitem
     *
     * @param workItem work item that will be associated with the new created message
     * @param message  Content of the message
     * @param msgType  Type of message {@link Severity}
     * @return Message object
     */
    public static MessageEvent createMessage(ProcessStepExecution workItem, String message, Severity msgType) {
        MessageEvent msg = new MessageEvent();

        // copy references to process, logs and workitem
        msg.setProcessId(workItem.getProcessId());
        msg.setNodeId(workItem.getNodeId());
        msg.setProcessInstanceId(String.valueOf(workItem.getProcessInstanceId()));
        msg.setCorrelationId(workItem.getCorrelationId());
        msg.setExecutionId(String.valueOf(workItem.getWorkItemId()));
        msg.setEventTime(new Date());
        msg.setMessageKey(message);
        msg.setSeverity(msgType);
        msg.setProcessEventType(EventType.MESSAGE_EVENT);
        msg.setDescription(message);

        return msg;
    }

    /**
     * Create err message for step message event.
     *
     * @param stepLog the step log
     * @return the message event
     */
    public static MessageEvent createErrMessageForStep(ProcessStepExecution stepLog) {
        return createMessageForStep(stepLog, Severity.ERROR);
    }

    /**
     * Create info message for step message event.
     *
     * @param stepLog the step log
     * @return the message event
     */
    public static MessageEvent createInfoMessageForStep(ProcessStepExecution stepLog) {
        return createMessageForStep(stepLog, Severity.INFO);
    }

    /**
     * Create message for step message event.
     *
     * @param workItem the work item
     * @param type     the type
     * @return the message event
     */
    public static MessageEvent createMessageForStep(ProcessStepExecution workItem, Severity type) {
        return createMessage(workItem, null, type);
    }
}
