package org.tkit.rhpam.quarkus.messaging;

import io.quarkus.arc.Arc;
import io.smallrye.mutiny.Uni;
import io.smallrye.reactive.messaging.amqp.AmqpMessage;
import io.vertx.mutiny.core.eventbus.EventBus;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.context.ManagedExecutor;
import org.eclipse.microprofile.reactive.messaging.Acknowledgment;
import org.eclipse.microprofile.reactive.messaging.Incoming;
import org.eclipse.microprofile.reactive.messaging.Message;
import org.tkit.rhpam.quarkus.messaging.model.AdditionalErrorInfo;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecution;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecutionResult;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

/**
 * The type Tkit rhpam wih service.
 */
@ApplicationScoped
@Slf4j
public class TkitRhpamWIHService {

    /**
     * The Process client service v 2.
     */
    @Inject
    ProcessClientServiceV2 processClientServiceV2;

    /**
     * The Bus.
     */
    @Inject
    EventBus bus;


    /**
     * Test 2 completion stage.
     *
     * @param message the message
     * @return the completion stage
     */
    @Transactional(Transactional.TxType.REQUIRED)
    @Acknowledgment(Acknowledgment.Strategy.MANUAL)
    @Incoming("jms.queue.TKIT_RHPAM_WIH")
    public CompletionStage<Void> onWIHMessage(AmqpMessage<String> message) {
        // TX 1 start

        // TX 2 start, TX 1 pause
        // if we fail here, whole execution fails. JMS message is returned to queue
        ProcessStepExecution execution = processClientServiceV2.startLog(message);
        // TX 2 end

        long startTime = System.currentTimeMillis();
        log.info("Process work item {} started.", execution);

        return CompletableFuture.runAsync(() -> {
            ProcessStepService stepService =
                    Arc.container().instance(ProcessStepService.class, new BusinessProcess.Literal(execution.getProcessId())).get();
            if (stepService == null) {
                log.error("No ProcessStepService for {}", execution.getProcessId());
            }
            AdditionalErrorInfo additionalErrorInfo = null;
            try {
                ProcessStepExecutionResult result = stepService.doExecute(execution);
                handleStepSuccess(execution, result, message);
            } catch (Exception e) {
                additionalErrorInfo = stepService.provideErrorInformation(execution, e);
                log.error("Business failed", e);
//                handleStepFailed(execution, e, message);
                message.nack(e);
            }

        });
    }

    private void handleStepFailed(ProcessStepExecution execution, Throwable t, Message<String> message) {
        log.info("#### handleStepFailed {}", t);
        try {
            processClientServiceV2.endLog(execution, null, message, t, null);
        } catch (Exception e) {
            log.error("EError in endlog", e);
        }
        message.nack(t);

        log.info("#### FINISHED handleStepFailed {}", t);
    }

    private void handleStepSuccess(ProcessStepExecution execution, ProcessStepExecutionResult processStepExecutionResult, Message<String> message) {
        log.info("#### handleStepSuccess {}", processStepExecutionResult);
        try {
            processClientServiceV2.endLog(execution, processStepExecutionResult, message, null, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
        log.info("### FINISH handleStepSuccess");
    }
}