package org.tkit.rhpam.quarkus.domain.daos;

import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.opentracing.Traced;
import org.hibernate.service.spi.ServiceException;

import org.tkit.quarkus.jpa.daos.AbstractDAO;
import org.tkit.rhpam.quarkus.domain.models.FailedStep;
import org.tkit.rhpam.quarkus.domain.models.FailedStepSearchCriteria;

import javax.enterprise.context.ApplicationScoped;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The type Failed step dao.
 */
@Slf4j
@ApplicationScoped
@Transactional
@Traced
public class FailedStepDAO extends AbstractDAO<FailedStep> {

    /**
     * Finds the failed steps by search criteria.
     *
     * @param searchCriteria the search criteria.
     * @return the corresponding list of failed steps
     * @throws ServiceException if the method fails.
     */
    public List<FailedStep> findBySearchCriteria(FailedStepSearchCriteria searchCriteria) {
        List<FailedStep> results;

            CriteriaBuilder cb = getEntityManager().getCriteriaBuilder();
            CriteriaQuery<FailedStep> cq = cb.createQuery(FailedStep.class).distinct(true);
            Root<FailedStep> root = cq.from(FailedStep.class);
            List<Predicate> predicates = new ArrayList<>();
            if (searchCriteria != null) {
                if (searchCriteria.getProcessInstanceId() != null) {
                    predicates.add(cb.equal(root.get("processInstanceId"), searchCriteria.getProcessInstanceId()));
                }
                if (searchCriteria.getParentProcessInstanceId() != null) {
                    predicates.add(cb.equal(root.get("parentProcessInstanceId"), searchCriteria.getParentProcessInstanceId()));
                }
                if (searchCriteria.getProcessName() != null) {
                    predicates.add(cb.equal(root.get("processName"), searchCriteria.getProcessName()));
                }
                if (searchCriteria.getProcessStepLogGuid() != null) {
                    predicates.add(cb.equal(root.get("processStepLogGuid"), searchCriteria.getProcessStepLogGuid()));
                }
                if (searchCriteria.getProcessStepName() != null) {
                    predicates.add(cb.equal(root.get("processStepName"), searchCriteria.getProcessStepName()));
                }
                if (searchCriteria.getReferenceBid() != null) {
                    predicates.add(cb.equal(root.get("referenceBid"), String.valueOf(searchCriteria.getReferenceBid())));
                }
                if (searchCriteria.getReferenceKey() != null) {
                    predicates.add(cb.equal(root.get("referenceKey"), searchCriteria.getReferenceKey()));
                }
                if (searchCriteria.getWorkItemId() != null) {
                    predicates.add(cb.equal(root.get("workItemId"), searchCriteria.getWorkItemId()));
                }
                if (searchCriteria.getFailureType() != null) {
                    predicates.add(cb.equal(root.get("failureType"), searchCriteria.getFailureType()));
                }
                if (searchCriteria.getStatus() != null) {
                    predicates.add(cb.equal(root.get("status"), searchCriteria.getStatus()));
                }
            }
            if (!predicates.isEmpty()) {
                cq.where(cb.and(predicates.toArray(new Predicate[0])));
            }
            cq.orderBy(cb.desc(root.get("creationDate")));
            results = getEntityManager().createQuery(cq).getResultList();

        return results;
    }

    /**
     * Gets process instance id to has failed step map by instance id list.
     *
     * @param processInstanceIds the process instance ids
     * @return the process instance id to has failed step map by instance id list
     */
    public Map<Long, Boolean> getProcessInstanceIdToHasFailedStepMapByInstanceIdList(List<Long> processInstanceIds)  {


            if (processInstanceIds == null || processInstanceIds.isEmpty()) {
                return new HashMap<>();
            }

            CriteriaBuilder cb = getEntityManager().getCriteriaBuilder();
            CriteriaQuery<Object[]> cq = cb.createQuery(Object[].class);
            Root<FailedStep> failedStepRoot = cq.from(FailedStep.class);

            cq.multiselect(failedStepRoot.get("processInstanceId"), cb.count(failedStepRoot));

            cq.where(failedStepRoot.get("processInstanceId").in(processInstanceIds), cb.equal(failedStepRoot.get("status"), FailedStep.FailedStepStatus.OPEN));
            cq.groupBy(failedStepRoot.get("processInstanceId"));

            List<Object[]> results = getEntityManager().createQuery(cq).getResultList();
            Map<Long, Boolean> ret = new HashMap<>();
            for (Object[] result : results) {
                Long processInstanceId = (Long) result[0];
                int count = ((Number) result[1]).intValue();
                ret.put(processInstanceId, count > 0);
            }

            return ret;

    }
}
