/*
 * Copyright (c) 2018, 1000kit.org, and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.tkit.rhpam.quarkus.messaging;

import ext.api.centrallog.api.ProcessLogEventEmitter;
import ext.api.centrallog.model.*;
import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.Tracer;
import io.smallrye.reactive.messaging.amqp.AmqpMessage;
import io.smallrye.reactive.messaging.annotations.Blocking;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.context.ManagedExecutor;
import org.eclipse.microprofile.context.ThreadContext;
import org.eclipse.microprofile.reactive.messaging.Acknowledgment;
import org.eclipse.microprofile.reactive.messaging.Incoming;
import org.hibernate.service.spi.ServiceException;
import org.tkit.rhpam.quarkus.domain.daos.DomainProcessInfoDAO;
import org.tkit.rhpam.quarkus.domain.daos.FailedStepDAO;
import org.tkit.rhpam.quarkus.domain.models.DomainProcessInfo;
import org.tkit.rhpam.quarkus.domain.models.FailedStep;
import org.tkit.rhpam.quarkus.domain.models.FailedStepSearchCriteria;
import org.tkit.rhpam.quarkus.domain.models.StageFlag;
import org.tkit.rhpam.quarkus.domain.models.enums.ProcessStepStatus;
import org.tkit.rhpam.quarkus.messaging.common.Constants;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.messaging.model.NodeType;
import org.tkit.rhpam.quarkus.messaging.model.ProcessEventItem;
import org.tkit.rhpam.quarkus.messaging.model.ProcessEventType;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecution;
import org.tkit.rhpam.quarkus.tracing.TraceFromMessage;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

import static org.tkit.rhpam.quarkus.messaging.common.MessageUtil.*;


/**
 * The process log executor listener.
 */
@Slf4j
@ApplicationScoped
public class ProcessEventListener {

    private static final String PROCESS_FAIL_ON_LAST_STEP_FAIL_ENABLED_PROPERTY = "ORG_TKIT_RHPAM_ENABLE_PROCESS_FAIL_ON_LAST_STEP_FAIL";
    @Inject
    ProcessLogEventEmitter centralEventEmitter;
    @Inject
    FailedStepDAO failedStepDAO;
    @Inject
    DomainProcessInfoDAO domainProcessInfoDAO;
    @Inject
    Tracer configuredTracer;
    @Inject
    ThreadContext threadContext;
    @Inject
    ManagedExecutor managedExecutor;

    /**
     * Execute the process event in the application.
     *
     * @param message the message from the process engine.
     * @return completion stage
     */
    @Transactional(Transactional.TxType.REQUIRED)
    @Acknowledgment(Acknowledgment.Strategy.MANUAL)
    @Incoming("tkitRhpamJbpmEvents")
    @TraceFromMessage
    @Blocking
    public CompletionStage<Void> onProcessEvent(AmqpMessage<String> message) {
        Span span = configuredTracer.activeSpan().setOperationName("onProcessEvent");
        try {
                ProcessEventItem logItem = createProcessLogItem(message);
                span.setOperationName("onProcessEvent - " + logItem.getProcessEventType());
                span.setTag("name", logItem.getName());
                log.info("Process Event data: {}", message.getApplicationProperties());
                // execute the work item
                switch (logItem.getProcessEventType()) {
                    case PROCESS_START:
                        startProcessEvent(logItem);
                        break;
                    case PROCESS_END:
                        endProcessEvent(logItem);
                        break;
                    case NODE_END:
                        endNodeEvent(logItem);
                        break;
                    case NODE_START:
                        startNodeEvent(logItem);
                        break;
                    default:
                        return message.nack(new IllegalArgumentException(String.format("ProcessEventType %s not known.", logItem.getProcessEventType())));
                }
                return message.ack();
            } catch (Exception ex) {
                log.error("Error execute the log item ", ex);
                return message.nack(ex);
            }
    }

    /**
     * Creates the process work item from the process engine message.
     *
     * @param msg the message from the process engine.
     * @return the corresponding process work item.
     * @throws Exception if the reading of the message attributes fails.
     */
    private ProcessEventItem createProcessLogItem(AmqpMessage<String> msg) throws Exception {

        // get the header information
        Long parentProcessInstanceId = getProperty(msg, PROP_PARENT_PROCESS_INSTANCE_ID);
        String parentProcessId = getProperty(msg, PROP_PARENT_PROCESS_ID);
        // requires tkit-rhpam >= 1.16
        Long subProcessId = getProperty(msg, PROP_TKIT_SUB_PROCESS_ID);
        Long processInstanceId = getProperty(msg, PROP_PROCESS_INSTANCE_ID);
        String processId = getProperty(msg, PROP_PROCESS_ID);
        String processVersion = getProperty(msg, PROP_PROCESS_VERSION);
        String listenerType = getProperty(msg, PROP_KIE_LISTENER_TYPE);
        String nodeName = getProperty(msg, PROP_KIE_NODE_NAME);
        Long nodeId = getProperty(msg, PROP_KIE_NODE_ID);
        String deploymentId = getProperty(msg, PROP_DEPLOYMENT_ID);
        Long referenceBid = getProperty(msg, PROP_REFERENCE_BID);
        String referenceKey = getProperty(msg, PROP_REFERENCE_KEY);
        String processLogGuid = getProperty(msg, PROP_PROCESS_LOG_GUID);
        String processName = getProperty(msg, PROP_PROCESS_NAME);
        Long workItemId = getProperty(msg, PROP_WORK_ITEM_ID);
        Long timerId = getProperty(msg, PROP_TIMER_ID);
        Long executionId = getProperty(msg, PROP_EXECUTION_ID);

        // the process event
        ProcessEventType processEvent = getEnumProperty(msg, PROP_KIE_PROCESS_LOG_EVENT, ProcessEventType.class);
        NodeType nodeType = getEnumProperty(msg, PROP_KIE_NODE_TYPE, NodeType.class);

        // load the parameters
        String body = msg.getPayload();
        Map<String, Object> parameters = deserializeBody(msg);

        // parsing the execution date
        Date executionDate = getStringDateProperty(msg, PROP_KIE_EXECUTION_DATE, new Date());

        Boolean processCompleteError = getProperty(msg, PROP_PROCESS_ERROR);
        String processCompleteOutcome = getProperty(msg, PROP_PROCESS_OUTCOME);
        String nodeResolutionStatus = getProperty(msg, PROP_TKIT_RESOLUTION_STATUS);
        String correlationId = getProperty(msg, PROP_TKIT_CORRELATION_ID);
        // create process work item
        return ProcessEventItem.builder()
                .nodeType(nodeType)
                .executionId(executionId)
                .timerId(timerId)
                .processLogGuid(processLogGuid)
                .body(body)
                .processEventType(processEvent)
                .deploymentId(deploymentId)
                .parentProcessInstanceId(parentProcessInstanceId)
                .parentProcessId(parentProcessId)
                .subProcessId(subProcessId)
                .processInstanceId(processInstanceId)
                .processId(processId)
                .processName(processName)
                .processVersion(processVersion)
                .listenerType(listenerType)
                .executionDate(executionDate)
                .name(nodeName)
                .nodeId(nodeId)
                .workItemId(workItemId)
                .referenceKey(referenceKey)
                .referenceBid(referenceBid)
                .parameters(parameters)
                .outcome(processCompleteOutcome)
                .nodeResolutionStatus(nodeResolutionStatus)
                .correlationId(correlationId)
                .error(processCompleteError != null ? processCompleteError : false)
                .build();
    }


    /**
     * Execute the start node event.
     *
     * @param logItem the item.
     * @throws ServiceException if the method fails.
     */
    private void startNodeEvent(ProcessEventItem logItem) throws ServiceException, RhpamException {
        NodeStartEvent event = new NodeStartEvent();
        event.setNodeId(String.valueOf(logItem.getNodeId()));
        event.setProcessEventType(EventType.NODE_START_EVENT);
        event.setExecutionId(String.valueOf(logItem.getExecutionId()));
        event.setEventTime(logItem.getExecutionDate());
        event.setCorrelationId(String.valueOf(logItem.getWorkItemId()));
        event.setProcessId(logItem.getProcessId());
        event.setProcessInstanceId(String.valueOf(logItem.getProcessInstanceId()));
        event.setBusinessRelevant(true);
        event.setNodeName(logItem.getName());

        // TODO - nodeTypes, metadata
        if (logItem.getNodeType() != null) {
            switch (logItem.getNodeType()) {
                case TIMER:
                    event.setNodeType(ext.api.centrallog.model.NodeType.TIMER);
                    break;
                case EVENT:
                    event.setNodeType(ext.api.centrallog.model.NodeType.EVENT);
                    break;
                case SUB_PROCESS:
                    event.setNodeType(ext.api.centrallog.model.NodeType.SUB_PROCESS);
                case END:
                    // ignore: is only for the process start and process end log item
                    break;
                default:
                    event.setNodeType(ext.api.centrallog.model.NodeType.ACTIVITY);
                    log.warn("Not supported node type: {} for log item {}. Using default ACTIVITY type.", logItem.getNodeType(), logItem);
            }
        }
        //Update domain process state
        DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(logItem.getReferenceKey(), String.valueOf(logItem.getReferenceBid()));
        if (dpi != null && logItem.getParentProcessId() == null) {
            //only update if we are a step of the top level process
            dpi.setCurrentProcessInstanceId(logItem.getProcessInstanceId());
            dpi.setCurrentProcessStepName(logItem.getName());
            if (dpi.getCurrentProcessStatus() == DomainProcessInfo.ProcessStatus.ERROR || dpi.getCurrentProcessStatus() == DomainProcessInfo.ProcessStatus.PENDING) {
                dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.RUNNING);
            }
            dpi.setCurrentProcessStepStatus(ProcessStepStatus.IN_EXECUTION.name());
            domainProcessInfoDAO.update(dpi);
        }

        centralEventEmitter.emitProcessEvent(event);
        log.info("startNodeEvent : {}", logItem);
    }

    /**
     * Execute the end node event.
     *
     * @param logItem the log item.
     * @throws ServiceException if the method fails.
     */
    private void endNodeEvent(ProcessEventItem logItem) throws ServiceException, RhpamException {
        //JBPM has confirmed node execution, we can notify log
        NodeEndEvent event = new NodeEndEvent();
        event.setNodeId(String.valueOf(logItem.getNodeId()));
        event.setProcessEventType(EventType.NODE_END_EVENT);
        event.setExecutionId(String.valueOf(logItem.getExecutionId()));
        event.setEventTime(logItem.getExecutionDate());
        event.setCorrelationId(logItem.getCorrelationId());
        event.setProcessId(logItem.getProcessId());
        event.setProcessInstanceId(String.valueOf(logItem.getProcessInstanceId()));
        event.setBusinessRelevant(true);
        event.setVariables(logItem.getParameters());
        if (logItem.getNodeResolutionStatus() != null) {
            event.setResolution(Resolution.fromValue(logItem.getNodeResolutionStatus()));
        }
        // If we are ending a node, that represents subprocess, try to check, if it has failed step and if so, mark it as failed
        if (logItem.getNodeType() == NodeType.SUB_PROCESS) {
            FailedStepSearchCriteria failedStepSearchCriteria = new FailedStepSearchCriteria();
            //we look for failed steps where current process is parent, and process name matches our node name
            failedStepSearchCriteria.setProcessName(logItem.getName());
            failedStepSearchCriteria.setProcessInstanceId(logItem.getSubProcessId());
            failedStepSearchCriteria.setParentProcessInstanceId(String.valueOf(logItem.getProcessInstanceId()));
            List<FailedStep> fsResults = failedStepDAO.findBySearchCriteria(failedStepSearchCriteria);
            if (fsResults != null && fsResults.size() > 0) {
                FailedStep failedStep = fsResults.get(0);
                event.setResolution(Resolution.FAILED);
                event.getMetadata().put("subProcessId", logItem.getSubProcessId());
                event.getMetadata().put("failedStepId", failedStep.getId());
                event.getMetadata().put("failedProcessStepName", failedStep.getProcessStepName());
                event.getMetadata().put("errorCode", failedStep.getErrorCode());
                event.getMetadata().put("errorMessage", failedStep.getErrorMessage());
                FailedStep parentFailedStepCopy = new FailedStep();
                parentFailedStepCopy.setParentProcessInstanceId(logItem.getParentProcessInstanceId() != null ?
                        String.valueOf(logItem.getParentProcessInstanceId()) : null);
                parentFailedStepCopy.setNodeId(String.valueOf(logItem.getNodeId()));
                //if workitem is null, set the execution id = which in such case will most likely be nodeid
                parentFailedStepCopy.setWorkItemId(logItem.getWorkItemId() != null ? logItem.getWorkItemId() : logItem.getExecutionId());
                parentFailedStepCopy.setStatus(FailedStep.FailedStepStatus.OPEN);
                parentFailedStepCopy.setNodeName(logItem.getName());
                parentFailedStepCopy.setProcessStepName(logItem.getName());
                parentFailedStepCopy.setReferenceKey(logItem.getReferenceKey());
                parentFailedStepCopy.setReferenceBid(logItem.getReferenceBid() != null ? String.valueOf(logItem.getReferenceBid()) : null);
                parentFailedStepCopy.setExecutionCount(1);
                parentFailedStepCopy.setFailureType(failedStep.getFailureType());
                parentFailedStepCopy.setErrorCode(failedStep.getErrorCode());
                parentFailedStepCopy.setProcessInstanceId(logItem.getProcessInstanceId());
                parentFailedStepCopy.setProcessId(logItem.getProcessId());
                parentFailedStepCopy.setDeploymentId(logItem.getDeploymentId());
                parentFailedStepCopy.setErrorMessage(failedStep.getErrorMessage());
                parentFailedStepCopy.setOriginalJMSMessage(failedStep.getOriginalJMSMessage());
                parentFailedStepCopy.setHeaders(failedStep.getHeaders());
                parentFailedStepCopy.setProcessName(logItem.getProcessName());
                failedStepDAO.create(parentFailedStepCopy);
            }
        }
        //update domain process state
        //only update if we are a step of the top level process
        if (logItem.getParentProcessId() == null) {
            DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(logItem.getReferenceKey(), String.valueOf(logItem.getReferenceBid()));
            if (dpi != null) {

                dpi.setCurrentProcessInstanceId(logItem.getProcessInstanceId());
                dpi.setCurrentProcessStepName(logItem.getName());
                if (event.getResolution() == Resolution.FAILED) {
                    dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.ERROR);
                    dpi.setCurrentProcessStepStatus(DomainProcessInfo.ProcessStatus.ERROR.name());
                } else {
                    if (dpi.getCurrentProcessStatus() == DomainProcessInfo.ProcessStatus.ERROR) {
                        dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.RUNNING);
                    }
                    dpi.setCurrentProcessStepStatus(ProcessStepStatus.COMPLETED.name());
                }
                domainProcessInfoDAO.update(dpi);
            }
        }

        centralEventEmitter.emitProcessEvent(event);
    }

    private boolean isTopLevelProcess(ProcessEventItem execution) {
        return execution.getParentProcessInstanceId() == null || execution.getParentProcessInstanceId() <= 0L;
    }

    /**
     * Execute the start process event.
     * Start the process log (sub-process, event, ...) or update existing process log start by the application.
     *
     * @param logItem the log item.
     * @throws ServiceException if the method fails.
     */
    private void startProcessEvent(ProcessEventItem logItem) throws ServiceException, RhpamException {
        ProcessStartEvent event = new ProcessStartEvent();

        event.setBusinessRelevant(true);
        event.setVariables(logItem.getParameters());
        event.setCorrelationId(String.valueOf(logItem.getProcessInstanceId()));
        event.setEventTime(new Date());
        // if this is subprocess, then it will not have any process_log_guid and we need alternative exec id
        String executionId = logItem.getProcessLogGuid() != null ? logItem.getProcessLogGuid() : String.valueOf(logItem.getProcessInstanceId());
        event.setExecutionId(executionId);
        event.getMetadata().put("nodeId", logItem.getNodeId());
        event.getMetadata().put(Constants.PROCESS_LOG_TENANT_ID, logItem.getParameters().get(Constants.PROCESS_LOG_TENANT_ID));
        event.getMetadata().put(Constants.PROCESS_LOG_TENANT_LOCATION_ID, logItem.getParameters().get(Constants.PROCESS_LOG_TENANT_LOCATION_ID));
        event.setProcessEventType(EventType.PROCESS_START_EVENT);
        event.setProcessId(logItem.getProcessId());
        event.setProcessInstanceId("" + logItem.getProcessInstanceId());
        event.setBusinessKey("" + logItem.getReferenceBid());
        event.setBusinessKeyType(logItem.getReferenceKey());
        event.setDeploymentId(logItem.getDeploymentId());
        event.setParentProcessInstanceId("" + logItem.getParentProcessInstanceId());
        event.setParentProcessId(logItem.getParentProcessId());
        event.setProcessName(logItem.getProcessName());
        event.setProcessVersion(logItem.getProcessVersion());
        DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(event.getBusinessKeyType(),
                event.getBusinessKey());
        if (dpi != null) {
            //if we are toplevel, update DPI
            if (isTopLevelProcess(logItem)) {
                dpi.setCurrentProcessInstanceId(logItem.getProcessInstanceId());
                dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.RUNNING);
            }

            StageFlag sf = new StageFlag();
            sf.setStatus(DomainProcessInfo.ProcessStatus.RUNNING);
            sf.setDate(new Date());
            sf.setInfo("" + logItem.getProcessInstanceId());
            dpi.getStageFlags().put(logItem.getProcessId(), sf);
            domainProcessInfoDAO.update(dpi);
        }

        centralEventEmitter.emitProcessEvent(event);
    }

    /**
     * Updates the process log by end process event.
     *
     * @param logItem the log item.
     * @throws ServiceException if the method fails.
     */
    private void endProcessEvent(ProcessEventItem logItem) throws ServiceException, RhpamException {
        ProcessEndEvent event = new ProcessEndEvent();

        event.setBusinessRelevant(true);
        event.setVariables(logItem.getParameters());
        event.setCorrelationId(String.valueOf(logItem.getProcessInstanceId()));
        event.setEventTime(new Date());

        // if this is subprocess, then it will not have any process_log_guid and we need alternative exec id
        String executionId = logItem.getProcessLogGuid() != null ? logItem.getProcessLogGuid() : String.valueOf(logItem.getProcessInstanceId());
        event.setExecutionId(executionId);

        event.setProcessEventType(EventType.PROCESS_END_EVENT);
        event.setProcessId(logItem.getProcessId());
        event.setProcessInstanceId("" + logItem.getProcessInstanceId());
        event.getMetadata().put("outcome", logItem.getOutcome());
        if (logItem.isError()) {
            //if the process has reached ERROR end node, we create a process level failed Step
            event.setResolution(Resolution.FAILED);
            FailedStep failedStep = createProcessLevelFailedStep(logItem);
            event.getMetadata().put("failedStepId", failedStep.getId());
            event.getMetadata().put("errorCode", failedStep.getErrorCode());
            event.getMetadata().put("errorMessage", failedStep.getErrorMessage());
        } else {
            event.setResolution(Resolution.SUCCESSFUL);
            // if this flag is true, and there is active failed step, we fail the process
            if (Boolean.parseBoolean(System.getenv(PROCESS_FAIL_ON_LAST_STEP_FAIL_ENABLED_PROPERTY))) {
                FailedStepSearchCriteria criteria = new FailedStepSearchCriteria();
                criteria.setProcessName(logItem.getProcessName());
                criteria.setStatus(FailedStep.FailedStepStatus.OPEN);
                criteria.setProcessInstanceId(logItem.getProcessInstanceId());
                List<FailedStep> failedStepList = failedStepDAO.findBySearchCriteria(criteria);
                if (failedStepList != null && failedStepList.size() == 1) {
                    FailedStep failedStep = failedStepList.get(0);
                    log.info("There is an active FailedStep for this process, marking it as FAILED");
                    event.setResolution(Resolution.FAILED);
                    event.getMetadata().put("failedStepId", failedStep.getId());
                    event.getMetadata().put("failedProcessStepName", failedStep.getProcessStepName());
                    event.getMetadata().put("errorCode", failedStep.getErrorCode());
                    event.getMetadata().put("errorMessage", failedStep.getErrorMessage());
                }
            }
        }

        DomainProcessInfo dpi = domainProcessInfoDAO.selectForUpdate(logItem.getReferenceKey(),
                String.valueOf(logItem.getReferenceBid()));
        if (dpi != null) {
            //if I am toplevel, then resolve
            if (isTopLevelProcess(logItem)) {
                dpi.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.COMPLETED);
                dpi.setProcessActive(false);
            }

            StageFlag sf = new StageFlag();
            sf.setStatus(event.getResolution() == Resolution.FAILED ? DomainProcessInfo.ProcessStatus.ERROR :
                    DomainProcessInfo.ProcessStatus.COMPLETED);
            sf.setDate(new Date());
            sf.setInfo("" + logItem.getProcessInstanceId());
            dpi.getStageFlags().put(logItem.getProcessId(), sf);
            domainProcessInfoDAO.update(dpi);
        }

        centralEventEmitter.emitProcessEvent(event);

    }

    private FailedStep createProcessLevelFailedStep(ProcessEventItem logItem) {

        FailedStep failedStep = new FailedStep();
        failedStep.setFailureType(FailedStep.FailureType.BUSINESS_ERROR);
        failedStep.setExecutionCount(1);
        failedStep.setProcessInstanceId(logItem.getProcessInstanceId());
        failedStep.setErrorCode(logItem.getOutcome());
        failedStep.setNodeId(String.valueOf(logItem.getNodeId()));
        failedStep.setNodeName(logItem.getProcessName());
        failedStep.setProcessId(logItem.getProcessId());
        failedStep.setParentProcessInstanceId(logItem.getParentProcessInstanceId() != null ?
                String.valueOf(logItem.getParentProcessInstanceId()) : null);
        failedStep.setProcessName(logItem.getProcessName());
        failedStep.setStatus(FailedStep.FailedStepStatus.OPEN);
        failedStep.setReferenceBid(String.valueOf(logItem.getReferenceBid()));
        failedStep.setReferenceKey(logItem.getReferenceKey());
        failedStep.setDeploymentId(logItem.getDeploymentId());
        return failedStepDAO.create(failedStep);
    }
}
