package org.tkit.rhpam.quarkus.rs;

import ext.api.centrallog.model.EventType;
import ext.api.centrallog.model.ProcessEndEvent;
import ext.api.centrallog.model.Resolution;
import lombok.extern.slf4j.Slf4j;
import ext.api.centrallog.api.ProcessLogEventEmitter;
import org.eclipse.microprofile.config.inject.ConfigProperty;
import org.eclipse.microprofile.openapi.annotations.tags.Tag;
import org.eclipse.microprofile.rest.client.inject.RestClient;
import org.tkit.rhpam.quarkus.domain.daos.DomainProcessInfoDAO;
import org.tkit.rhpam.quarkus.domain.models.DomainProcessInfo;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.rs.clients.KieServerRestClient;
import org.tkit.rhpam.quarkus.rs.model.ProcessInstanceDTO;
import org.tkit.rhpam.quarkus.rs.model.ProcessInstanceRemovalRequest;
import org.tkit.rhpam.quarkus.rs.model.ProcessInstanceWrapper;

import javax.enterprise.context.RequestScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import javax.ws.rs.*;
import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

@RequestScoped
@Transactional
@Path("/processInstances")
@Tag(name = "processInstanceAPI")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
@Slf4j
public class ProcessInstanceRestController {

    @Inject
    private ProcessLogEventEmitter processLogEventEmitter;

    @Inject
    private DomainProcessInfoDAO domainProcessInfoDAO;

    @Inject
    @RestClient
    private KieServerRestClient kieServerRestClient;

    @ConfigProperty(name = "kie.admin.user")
    private Optional<String> usernameParam;

    @ConfigProperty(name = "kie.admin.pwd")
    private Optional<String> passwordParam;


    @POST
    @Path("/{processInstanceId}/abortRequest")
    public Response abortProcess(@PathParam("processInstanceId") Long processInstanceId, ProcessInstanceRemovalRequest removalRequest) throws RhpamException {
        String username = usernameParam.map(Object::toString).orElseThrow(() -> new RhpamException("Kie server username can not be null. " +
                "Please provide property kie.admin.user with correct value", Response.Status.BAD_REQUEST));
        String password = passwordParam.map(Object::toString).orElseThrow(() -> new RhpamException("Kie server password can not be null." +
                "Please provide property kie.admin.pwd with correct value", Response.Status.BAD_REQUEST));
        String authentication = getClientCredentialsAuth(username, password);
        cleanDomainProcessInfo(processInstanceId, removalRequest.isDeleteDomainProcessInfo());
        abortMainInstance(authentication, processInstanceId, removalRequest);
        abortChildInstances(authentication, processInstanceId, removalRequest);
        return Response.accepted().build();
    }

    private void abortMainInstance(String authentication, Long processInstanceId, ProcessInstanceRemovalRequest removalRequest) throws RhpamException {
        ProcessInstanceDTO processInstance;
        try {
            processInstance = kieServerRestClient.getProcessInstance(authentication, removalRequest.getContainerId(), processInstanceId)
                    .readEntity(ProcessInstanceDTO.class);
        } catch (Exception exception) {
            throw new RhpamException("Process instance with id " + processInstanceId + " does not exist.", Response.Status.NOT_FOUND);
        }
        if (processInstance.getState() == 1) {
            kieServerRestClient.abortProcessInstance(authentication, processInstance.getContainerId(), processInstanceId);
            sendProcessEndEvent(processInstanceId, removalRequest.getProcessId(), removalRequest.getMessage());
        }
    }

    private void abortChildInstances(String authentication, Long processInstanceId, ProcessInstanceRemovalRequest removalRequest) throws RhpamException {
        List<ProcessInstanceDTO> childInstances = kieServerRestClient.getProcessInstances(authentication, removalRequest.getContainerId(), processInstanceId)
                .readEntity(new GenericType<ProcessInstanceWrapper>() {
                }).getProcessInstances()
                .stream()
                .filter(instance -> instance.getState() == 1)
                .collect(Collectors.toList());
        List<Long> processInstanceIdsToAbort = childInstances
                .stream()
                .map(ProcessInstanceDTO::getId)
                .collect(Collectors.toList());
        if (!processInstanceIdsToAbort.isEmpty()) {
            kieServerRestClient.abortProcessInstances(authentication, removalRequest.getContainerId(), processInstanceIdsToAbort);
            for (ProcessInstanceDTO processInstanceDTO : childInstances) {
                sendProcessEndEvent(processInstanceDTO.getId(), processInstanceDTO.getProcessId(), removalRequest.getMessage());
            }
        }
    }

    private void cleanDomainProcessInfo(Long processInstanceId, boolean deleteDomainProcessInfo) {
        DomainProcessInfo domainProcessInfo = domainProcessInfoDAO.selectForUpdate(processInstanceId);
        if (domainProcessInfo != null) {
            if (deleteDomainProcessInfo) {
                domainProcessInfoDAO.delete(domainProcessInfo);
            } else {
                domainProcessInfo.setCurrentProcessStatus(DomainProcessInfo.ProcessStatus.ABORTED);
                domainProcessInfo.getStageFlags().forEach((s, stageFlag) -> {
                    if (DomainProcessInfo.ProcessStatus.RUNNING.equals(stageFlag.getStatus())) {
                        stageFlag.setStatus(DomainProcessInfo.ProcessStatus.ABORTED);
                    }
                });
                domainProcessInfoDAO.update(domainProcessInfo);
            }
        } else {
            log.warn("Domain process info with process instance id {} not found.", processInstanceId);
        }
    }

    private void sendProcessEndEvent(Long processInstanceId, String processId, String message) throws RhpamException {
        try {
            ProcessEndEvent processEndEvent = new ProcessEndEvent();
            processEndEvent.executionId(processInstanceId.toString());
            processEndEvent.processInstanceId(processInstanceId.toString());
            processEndEvent.correlationId(processInstanceId.toString());
            processEndEvent.processId(processId);
            processEndEvent.setResolution(Resolution.ABORTED);
            processEndEvent.processEventType(EventType.PROCESS_END_EVENT);
            processEndEvent.setBusinessRelevant(true);
            processEndEvent.setMetadata(Map.of("Message", message));
            processEndEvent.setEventTime(new Date());
            processLogEventEmitter.emitProcessEvent(processEndEvent);
        } catch (Exception e) {
            throw new RhpamException("Failed to send PROCESS_END_EVENT on abort", Response.Status.INTERNAL_SERVER_ERROR);
        }
    }

    private String getClientCredentialsAuth(String username, String password) {
        String tmp = username + ":" + password;
        return "Basic " + Base64.getEncoder().encodeToString(tmp.getBytes(StandardCharsets.UTF_8));
    }

}
