package org.tkit.rhpam.quarkus.messaging;

import ext.api.centrallog.model.Severity;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.microprofile.opentracing.Traced;
import org.tkit.quarkus.log.cdi.LogService;
import org.tkit.rhpam.quarkus.common.MessageKeys;
import org.tkit.rhpam.quarkus.domain.models.FailedStep;
import org.tkit.rhpam.quarkus.messaging.common.RhpamException;
import org.tkit.rhpam.quarkus.messaging.model.AdditionalErrorInfo;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecution;
import org.tkit.rhpam.quarkus.messaging.model.ProcessStepExecutionResult;
import org.tkit.rhpam.quarkus.messaging.model.ResolutionStatus;
import org.tkit.rhpam.quarkus.process.ProcessStepService;

import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;

import static org.tkit.rhpam.quarkus.messaging.common.MessageUtil.*;

@ApplicationScoped
@Slf4j
@Traced
public class BusinessErrorHandlerStepService implements ProcessStepService {

    @Inject
    FailedStepService failedStepService;

    @Override
    @LogService(stacktrace = false)
    public ProcessStepExecutionResult doExecute(ProcessStepExecution workItem) throws RhpamException {

        String referencedStepName = workItem.getParameter(PROP_BEH_REFERENCED_STEP_NAME);
        String resolutionSignalId = workItem.getParameter(PROP_BEH_RESOLUTION_SIGNAL_NAME);
        String resolutionActions = workItem.getParameter(PROP_BEH_RESOLUTION_ACTIONS);
        if (referencedStepName == null || resolutionSignalId == null || resolutionActions == null) {
            log.error("Invalid business error handler definition! Please specify all of {referencedStepName,resolutionSignalId," +
                    "resolutionActions}");
        }
        // at this point, we have no info about the real step that failed, but we should have a prepared FailedStep in DB, so we update it
        FailedStep failedStep = failedStepService.updateFailedStep(workItem, referencedStepName, resolutionActions, resolutionSignalId);

        failedStepService.emitFailedStepEvent(failedStep);
        //This does not throw exception
//            supportToolSubmitterService.submitExternalSupportToolIssue(failedStep);
        return ProcessStepExecutionResult.builder().status(ResolutionStatus.SUCCESSFUL).build();
    }

    @Override
    public AdditionalErrorInfo provideErrorInformation(ProcessStepExecution workItem, Exception exc) {
        if (exc instanceof RhpamException) {
            // if the business err handler failed on the interim error caused by missign failed step from subprocess,
            // and we can still retry, we suppress the message
            if (((RhpamException) exc).getErrorCode().equals(MessageKeys.BUSINESS_ERR_HANDLER_MISSING_FAILED_STEP.getErrorCode())) {
                AdditionalErrorInfo aei = ProcessStepService.super.provideErrorInformation(workItem, exc);
                aei.setSuppressErrorMessage(AdditionalErrorInfo.ErrorSuppressionPolicy.IF_RETRY_ALLOWED);
                return aei;
            }
        }
        return ProcessStepService.super.provideErrorInformation(workItem, exc);
    }
}
